#
from pygpibtoolkit.pygpib import CommandRegister, Constants, Command
from pygpibtoolkit.pygpib import BoolValue, IntValue, FloatValue
from pygpibtoolkit.pygpib import PercentageValue, FrequencyValue, DurationValue
from pygpibtoolkit.pygpib import EnumValue, StringValue
from pygpibtoolkit.pygpib import Mode, ModeCommand

from pygpibtoolkit.gpibcontroller import AbstractGPIBDevice, deviceRegister

class Register(CommandRegister):
    """
    Register class for every HPIB commands for the HP356X devices
    """
# we must instanciate the singleton for registering mechanism to work
Register()     


#####################
# HP3562A constants and command set
# **VERY INCOMPLETE**

# GPIB buffer size is 3x80 characters lines
class STATUS_BYTE(Constants):
    # HP3562A Status Byte, as returned by a serial poll
    _constants = [(0x40, "RQS", "Request Service"), # when sent in response to a serial poll
                  (0x20, "ERR", "GPIB error"),
                  (0x10, "RDY", "ready to accept GPIB commands"),
                  ]

    conditions = [(0, "NSR", "No service requested"),
                  (1, "USRQ1", "User SRQ #1"),
                  (2, "USRQ1", "User SRQ #2"),
                  (3, "USRQ1", "User SRQ #3"),
                  (4, "USRQ1", "User SRQ #4"),
                  (5, "USRQ1", "User SRQ #5"),
                  (6, "USRQ1", "User SRQ #6"),
                  (7, "USRQ1", "User SRQ #7"),
                  (8, "USRQ1", "User SRQ #8"),
                  (9, "EOD", "End of disk action"),
                  (10, "EOP", "End of plot action"),
                  (11, "STCH", "Instrument status changed"), # any change in 
                  # the status register sets this condition
                  (12, "PWR", "Power on"),
                  (13, "KEY", "Key pressed"),
                  (14, "DCP", "Device clear plotter (listen)"),
                  # ...
                  ]
    def __init__(self):
        super(STATUS_BYTE, self).__init__()
        self._conditions = dict([(x[0], x[1]) for x in self.conditions])
        self._rev_conditions = dict([(x[1], x[0]) for x in self.conditions])
        self._long_conditions = dict([(x[0], x[2]) for x in self.conditions])
               
    def byte_condition(self, byte):
        byte = byte & 0x8F
        return self._conditions.get(byte, "N/A")

class IS(Flag):
    "INTERNAL STATUS - Internal Status Register"
    _constants = [(0x01, "MEASP", "measeurement pause"),
                  (0x02, "ASQP", "Auto sequence pause"),
                  (0X04, "EOM", "End of measurement, capture or throughput"),
                  (0x08, "EOAS", "End of auto sequence"),
                  (0x10, "SWPR", "Sweep point ready"),
                  (0x20, "CH1OV", "Channel 1 overrange"),
                  (0x40, "CH2OV", "Channel 2 overrange"),
                  (0X80, "CH1HR", "Channel 1 half range"),
                  (0x100, "CH2HR", "Channel 2 half range"),
                  (0x200, "SFALT", "Source fault"),
                  (0x400, "RUNL", "Reference unlock"),
                  (0x800, "RMKT", "Remote marker knob turn"),
                  (0x1000, "REKT", "Remote entry knob turn"),
                  (0x2000, "ASRC", "Asctive Status Register changed"),
                  (0x4000, "PWRF", "Power-on test failed"),
                  ]
    
class STA(Flag):
    "STATUS QUERY - Status/event query"
    _constants = [(0x01, "N/A", "Not used"),
                  (0x02, "N/A", "Not used"),
                  (0x04, "KEY", "Key pressed"),
                  (0x08, "N/A", "Not used"),
                  (0x10, "RDY", "Ready"),
                  (0x20, "ERR", "Error"),
                  (0x40, "RQS", "Request"),
                  (0x80, "MOS", "Message on screen"),
                  (0x100, "MEASP", "Measurement pause"),
                  (0x200, "ASQP", "Auto sequence pause"),
                  (0x400, "EOM", "End of measurement, capture or throughput"),
                  (0x800, "EOAS", "End of auto sequence"),
                  (0x1000, "SWPR", "Sweep point ready"),
                  (0x2000, "CH1OV", "Channel 1 overrange"),
                  (0x4000, "CH2OV", "Channel 2 overrange"),
                  (0x8000, "MAOV", "Math overflow"),
                  ]
class AS(Flag):
    "ACTIVITY STATUS - Activity Status Register"
    _constants = [(0x01, "CKFL", "Check fault log"),
                  (0x02, "FITR", "Filling time record"),
                  (0x04, "FLTR", "Filters settings"),
                  (0x08, "CFTP", "Curve fir in progress"),
                  (0x10, "MSSM", "Missed sample"),
                  (0x20, "TMPR", "Timed preview"),
                  (0x40, "ACDA", "Accept data"),
                  #...
                  ]


class TraceDisplay(Mode):
    "Trace Display"
    A = ModeCommand("A trace", "A")
    B = ModeCommand("B trace", "B")
    
class ARM(Command):
    "ARM - Arm"
class NAVG(IntValue):
    "Number of averages"

class AverageMode(Mode):
    "Average mode"
    _condition = lambda device: device.MeasMode in ['LINEAR RES','LOG RES', 'TIME CAPTUR']
    AVOF = ModeCommand("Average off", "AVG OFF")
    STBL = ModeCommand("Stable (means)", "STABLE (MEAN)")
    EXP  = ModeCommand("Exponential", "EXPON")
    PHLD = ModeCommand("Peak hold", "PEAK HOLD")
    CNPK = ModeCommand("Cont. peak", "CONT PEAK")

class TIAV(BoolValue):
    "TIM AV ON OFF - Time average"
    _condition = lambda device: device.MeasMode in ['LINEAR RES','TIME CAPTUR','SWEPT SINE']

class OVLP(PercentageValue):
    "OVRLP% - Overlap (%)"
    _condition = lambda device: device.MeasMode in ['LINEAR RES','TIME CAPTUR','SWEPT SINE']
    
class AUTO(BoolValue):
    "AUTO ON OFF - Auto calibration"

class SNGC(Command):
    "SINGLE CAL - Single calibration"

class CoordMode(Mode):
    "Coord mode"
    MGDB = ModeCommand("Mag. (dB)", "MAG (dB)")
    MDBM = ModeCommand("Mag. (dBm)", "MAG (dBm)")
    MGLG = ModeCommand("Mag. (log)", "MAG (LOG)")
    MAG  = ModeCommand("Mag. (lin)", "MAG (LIN)")
    PHSE = ModeCommand("Phase", "PHASE")
    REAL = ModeCommand("Real", "REAL")
    IMAG = ModeCommand("Imag.", "IMAG")
    NYQT = ModeCommand("Nyquist", "NYQUIST")
    NICL = ModeCommand("Nichol", "NICHOL")

# FREQ menu
class FRS(FrequencyValue):
    "FREQ SPAN - Freq. span"
class SF(FrequencyValue):
    "START FREQ - Start freq."
class CF(FrequencyValue):
    "CENTER FREQ - Center freq."
    _units = "hz","khz","mhz","ord","rmp"
    _condition = lambda device: device.MeasMode in ['LINEAR RES','TIME CAPTUR','SWEPT SINE']
class ZST(Command):
    "ZERO START - Zero start"
    _condition = lambda device: device.MeasMode in ['LINEAR RES','TIME CAPTUR','SWEPT SINE']
class MAXS(FrequencyValue):
    "MAX SPAN - Max span"
    _condition = lambda device: device.MeasMode in ['LINEAR RES','TIME CAPTUR','SWEPT SINE']
class TLN(DurationValue):
    "TIME LENGTH - Time len."
    _condition = lambda device: device.MeasMode in ['LINEAR RES','TIME CAPTUR','SWEPT SINE']
class ESMP(BoolValue):
    "E SMPL ON OFF - E sample"
    _condition = lambda device: device.MeasMode in ['LINEAR RES','TIME CAPTUR','SWEPT SINE']
class SMPF(FrequencyValue):
    "SAMPLE FREQ - Sample freq."
    _condition = lambda device: device.MeasMode in ['LINEAR RES','TIME CAPTUR','SWEPT SINE']
class SPF(FrequencyValue):
    "STOP FREQ - Stop freq."
    _condition = lambda device: device.MeasMode in ['SWEPT SINE']
class SWRT(FrequencyValue):
    "SWEEP RATE - Sweep rate"

class ABIB(Command):
    "ABORT HP-IB - Abort HPIB"

# INPUT COUPLING
class C1AC(EnumValue):
    "CHAN1 AC DC - Channel 1 AC/DC"
    _values = ["DC", "AC"]
    
class Chan1Coupling(Mode):
    "Channel 1 coupling"
    FLT1 = ModeCommand("Float", "FLOAT CHAN1")
    GND1 = ModeCommand("Ground", "GROUND CHAN1")

class C2AC(EnumValue):
    "CHAN2 AC DC - Channel 2 AC/DC"
    _values = ["AC", "DC"]
    
class Chan2Coupling(Mode):
    "Channel 2 coupling"
    FLT2 = ModeCommand("Float", "FLOAT CHAN2")
    GND2 = ModeCommand("Ground", "GROUND CHAN2")

class LCL(Command):
    "LOCAL - Local"

class C1RG(VoltageValue):
    "CHAN 1 RANGE - Channel 1 range"
    _units = "V","mV","Vrms","mVrms","dBV", "EU" 

class C2RG(VoltageValue):
    "CHAN 2 RANGE - Channel 2 range"
    _units = "V","mV","Vrms","mVrms","dBV", "EU" 

# MEAS MODE
class MeasMode(Mode):
    "Measurement mode"
    LNRS = ModeCommand("Linear resolution", "LINEAR RES")
    LGRS = ModeCommand("Log resolution", "LOG RES")
    SSIN = ModeCommand("Swept sinus", "SWEPT SINE")
    CPTR = ModeCommand("Time Capture", "TIME CAPTUR")
    
# MEAS DISP
class MeasDisp(Mode):
    # TODO
    pass

# SELECT MEAS
class SelectMeas(Mode):
    "Select measurement"
    FRSP = ModeCommand("Frequency response", "FREQ RESP", condition=lambda device: device.MeasMode in ["LINEAR RES", "LOG RES", "SWEPT SINE"])
    PSPC = ModeCommand("Power spectrum", "POWER SPEC", condition=lambda device: device.MeasMode in ["LINEAR RES","LOG RES","TIME CAPTUR"])
    AUCR = ModeCommand("Auto correlation", "AUTO CORR", condition=lambda device: device.MeasMode in ["LINEAR RES","TIME CAPTUR"])
    CCOR = ModeCommand("Cross correlation", "CROSS CORR", condition=lambda device: device.MeasMode == "LINEAR RES")
    HIST = ModeCommand("Histogramm", "HIST", condition=lambda device: device.MeasMode in ["LINEAR RES", "TIME CAPTUR"])
    CH12 = ModeCommand("Channel 1&2 active", "CH 1&2 ACTIVE", condition=lambda device: device.MeasMode in ["LINEAR RES", "LOG RES"])
    CH1 = ModeCommand("Channel 1 active", "CH 1 ACTIVE", condition=lambda device: device.MeasMode in ["LINEAR RES", "LOG RES", "TIME CAPTUR"])
    CH2 = ModeCommand("Channel 2 active", "CH 2 ACTIVE", condition=lambda device: device.MeasMode in ["LINEAR RES", "LOG RES", "TIME CAPTUR"])

class REV(StringValue):
    _readonly = True
    "REVISION - Revision"

class SACR(Command):
    "SACR - Send Auto Carrier"
    
class HP356XDevice(AbstractGPIBDevice):
    _accepts = ["HP3562A", "HP3563A"]
    _idn = "ID?"
    _cmd_register = Register
    def manage_srq(self, statusbyte):
        print "Managing SRQ", statusbyte
        
    
    
deviceRegister.register_manager(HP356XDevice)
