"""
gpib: create serial connection to GPIB-USB device (ProLogix is the
only supported device for now).
"""
import serial
from serial.serialutil import SerialException
import time
from pygpibtoolkit.tools import AbstractRegister 

class ConnectionError(Exception):
    pass

class Constants(object):
    def __init__(self):
        self.constants = {}
        self.descriptions = {}
        self.rev_constants = {}
        for v, k, m in self._constants:
            self.k = v
            self.constants[v] = k
            self.rev_constants[k] = v
            self.descriptions[v] = m
            
    def __getitem__(self, k):
        if isinstance(k, basestring):
            return self.rev_constants[k]
        else:
            return self.constants[k]

    def get_description(self, k):
        if isinstance(k, basestring):
            k = self.rev_constants[k]
        return self.descriptions[k]

class MODE(Constants):    
    _constants = [(1, "CONTROLLER", "Set device as Controller in Charge"),
                  (0, "DEVICE", "Set device as simple listener"),
                  ]

class ModeCommand(object):
    def __init__(self, description, name, condition=None):
        self.name = name
        self.description = description
        self.condition = condition

class CommandRegister(object):
    _instances = {}
    _registered_type = None #Command
    def __new__(cls):
        # implements a singleton *per class*
        if cls.__name__ not in cls._instances:
            if cls._registered_type is None:
                cls._registered_type = Command
            instance = super(CommandRegister, cls).__new__(cls)
            instance.registry = {}
            cls._instances[cls.__name__] = instance
        return cls._instances[cls.__name__]

    @classmethod
    def add(cls, registered_cls):
        for registry in cls._instances.values():
            if registry.__module__ == registered_cls.__module__:
                break
        else:
            return
        
        assert issubclass(registered_cls, registry._registered_type)
        if registered_cls is registry._registered_type:
            return

        name = registered_cls.__name__
        if name not in registry:
            registry.registry[name] = registered_cls
            
    def __contains__(self, key):
        return key in self.registry
    def keys(self):
        return self.registry.keys()
    def items(self):
        return self.registry.items()
    def values(self):
        return self.registry.values()
    def __getitem__(self, key):
        return self.registry[key]

class MetaCommand(type):
    """
    Metaclass for HPIB command. Used to register all commands.
    """
    _commands = {}
    def __init__(cls, name, bases, dct):
        # called at class creation
        super(MetaCommand, cls).__init__(name, bases, dct)
        if name not in [ "AbstractCommand","Command"]:
            CommandRegister().add(cls)
    
class AbstractCommand(object):
    """
    Base class for HPIB command descritption.

    This is actually a attribute descriptor, which should have
    AbstractGPIBDevice derived classes as owner.
    """
    __metaclass__ = MetaCommand
    _readonly = True
    _init_value = None
    
    def __get__(self, instance, owner):
        if instance is None:
            return self
        return instance._get(self.__class__.__name__)
    def __set__(self, instance, value):
        if instance is None:
            return self
        return instance._set(self.__class__.__name__, value)

class Command(AbstractCommand):
    def build_get_cmd(self):
        return self.__class__.__name__

    def build_set_cmd(self, *value):
        raise ValueError, "Can't set value for command '%s'"%self.__class__.__name__

    def convert_from(self, *value):
        return None
    
class AbstractValue(Command):
    _readonly = False
    def build_get_cmd(self):
        return self.__class__.__name__ + "?"

    def build_set_cmd(self, *value):
        value = self.convert_to(value)
        cmd = "%s %s"%(self.__class__.__name__, value)
        return cmd
    
    def convert_to(self, *value):
        if value:
            return str(value[0])
        return ""
    
    def convert_from(self, *value):
        if value:
            return self._type(value[0].strip())
        return None
    
class BoolValue(AbstractValue):
    _type = bool
    def convert_from(self, *value):
        if value:
            return value[0] and value[0].lower() in ['1','true','yes','on']
        return False
    
    def convert_to(self, *value):
        if value:
            return  value[0] and "1" or "0"
        return "" # XXX is it correct?
    
class IntValue(AbstractValue):
    _type = int
    def convert_from(self, *value):
        if value:
            # int is resturned as a string representing a float 
            return self._type(float(value[0].strip()))
        return None

class flag(int):
    def __new__(cls, val, constants):
        val = int.__new__(cls, val)
        val._constants = constants
        for v, name, desc in constants:
            if name not in ['', 'N/A']:
                setattr(val, name, v)
        return val
    
    def __str__(self):
        return '%s <' + '|'.join([x[1] for x in self._constants if x[0]&self]) + ">"
    def flags(self):
        return [x[1] for x in self._constants if x[0] & (self and x[1]) not in ['','N/A']]
        
    def descriptions(self):
        return [x[2] for x in self._constants if (x[0] & self) and x[1] not in ['','N/A']]

class Flag(IntValue):
    _readonly = True
    _constants = []
    _type = flag
    def convert_from(self, *value):
        if value:
            return self._type(float(value[0].strip()), _constants)
        return None
    
class FloatValue(AbstractValue):
    _type = float

class PercentageValue(FloatValue):
    pass #TODO

class FloatUnitValue(FloatValue):
    """
    A Float value with unit (frequency, etc.)
    """
    def convert_to(self, *value):
        if value:
            if len(value)==1:
                if isinstance(value[0], basestring):
                    value = value[0].strip().split()
                    if len(value)==1:
                        freq = float(value)
                        unit = self._units[0]
                    elif len(value)==2:
                        freq = float(value[0])
                        unit = value[1]
                    else:
                        raise ValueError, "Can't interpret %s as a %s"%(repr(value), self._name)
                else:
                    freq = float(value[0])
                    unit = self._units[0]
            elif len(value)==2:
                freq = float(value[0])
                unit = value[1]
            else:
                raise ValueError, "Can't interpret %s as a %s"%(repr(value), self._name)
            assert unit.lower() in self._units, "Unit is not correct (%s)"%unit
            return "%s%s"%(freq, unit)
        return "" # XXX is it correct?

class FrequencyValue(FloatUnitValue):
    _units = ['Hz','kHz','mHz',]
    _name = "frequency"

class VoltageValue(FloatUnitValue):
    _units = ['V','mV',]
    _name = "voltage"
    
class DurationValue(FloatUnitValue):
    _units = ['sec','msec','usec','min']
    _name = "duration"

class StringValue(AbstractValue):
    _type = str
    def convert_to(self, *value):
        if value:
            return "'%s'"%str(value[0])
        return "''"

class EnumValue(StringValue):
    _values = []
    def convert_to(self, *value):
        if value:
            assert value[0] in self._values
            return "%s"%self._values.index(value[0])
        return "" # XXX

class Mode(AbstractValue):
    pass

# TODO
# class STATUS_BYTE(Constants):
#     # IEEE 488.2 Status Byte constants
#     MAV = 0x10 # Message AVailable: bit 4 of the Status Byte
#     ESB = 0x20 # Event Status Bit: bit 5 of the Status Byte
#     MSS = 0x40 # Master Summary Status bit: bit 6 of the Status Byte (NOT
#                # sent in response to a serial poll)
#     RQS = 0x40 # Request Service: bit 6 of the Status Byte (when sent in
#                # response to a serial poll)
# class SESR(Constants):
#     # SESR constants (Standard Event Status Register)
#     PON = 0x80 # Power On: Power has been turned On since last register
#                # read access
#     URQ = 0x40 # User Request: the user has activated some device control
#                # (whatever the Remote Local state is)
#     CME = 0x20 # Command Error
#     EXE = 0x10 # Execution Error
#     DDE = 0x08 # Device Dependant Error
#     QYE = 0x04 # QuerY Error (attempt to read data while Output Queue is
#                # empty, or data in the OQ was lost)
#     RQC = 0x02 # Request Control: tell the CiC that the device wants to
#                # become the CiC
#     OPC = 0x01 # Operation Complete: device has completed any pending
#                # operation (ready to accept new commands). This bit is
#                # generated in response to a OPC command.
    
