# This program is free software; you can redistribute it and/or modify it under
# the terms of the GNU General Public License as published by the Free Software
# Foundation; either version 2 of the License, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details
#
# You should have received a copy of the GNU General Public License along with
# this program; if not, write to the Free Software Foundation, Inc.,
# 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
""" Copyright (c) 2007-2018 David Douard (Paris, FRANCE).
https://bitbucket.org/dddouard/pygpibtoolkit -- mailto:david.douard@sdfa3.org

"""
from pygpibtoolkit.pygpib import ModeCommand
from pygpibtoolkit.pygpib import VoltageValue, FrequencyValue
from pygpibtoolkit.pygpib import FloatUnitValue
from pygpibtoolkit.pygpib import SimpleMode

from pygpibtoolkit.gpibcontroller import AbstractGPIBDevice, deviceRegister


class _VoltageValue(VoltageValue):
    _units = "VL", "MV", "UV"


class APA(_VoltageValue):
    "CHAN A AMP - Channel A amplitude"
class APB(_VoltageValue):  # noqa
    "CHAN B AMP - Channel B amplitude"
class APC(_VoltageValue):  # noqa
    "CHAN C AMP - Channel C amplitude"
class APD(_VoltageValue):  # noqa
    "CHAN D AMP - Channel D amplitude"


class Backlight(SimpleMode):
    "Backlight"

    BF = ModeCommand("Off", "OFF")
    BO = ModeCommand("On", "ON")


class _PerOutputMode(SimpleMode):

    def __init__(self):
        self._cmds = []
        for cmdname, cmd in self.__class__.__dict__.items():
            if isinstance(cmd, ModeCommand):
                self._cmds.append(self._prefix + cmdname)

    def build_set_cmd(self, value):
        # XXX HERE: todo: make it work with value not in self._cmds
        # (eg. value=="AU", cmds == ['FS1AU')
        assert self._prefix+value in self._cmds
        return self._prefix+value, value


class Filter1(_PerOutputMode):
    "Filter for output 1"
    _prefix = "FS1"

    SH = ModeCommand("Elliptic (sharp cutoff)", "ELIPT.")
    LO = ModeCommand("Gaussian (low overshoot)", "GAUSS.")
    AU = ModeCommand("Automatic", "AUTO")


class Filter2(Filter1):
    "Filter for output 2"
    _prefix = "FS2"


class Float1(_PerOutputMode):
    "Float Control for output 1"
    _prefix = "FC1"
    ON = ModeCommand("On", "ON")
    OF = ModeCommand("Off", "OFF")


class Float2(Float1):
    "Float Control for output 2"
    _prefix = "FC2"


class _FrequencyValue(FrequencyValue):
    _units = ['HZ', 'KZ']


class FRA(_FrequencyValue):
    "CHAN A FRQ - Channel A frequency"
class FRB(_FrequencyValue):  # noqa
    "CHAN B FRQ - Channel B frequency"
class FRC(_FrequencyValue):  # noqa
    "CHAN C FRQ - Channel C frequency"
class FRD(_FrequencyValue):  # noqa
    "CHAN D FRQ - Channel D frequency"


class OutputControl1(_PerOutputMode):
    "Output 1"
    _prefix = "OO1"
    ON = ModeCommand("On", "ON")
    OF = ModeCommand("Off", "OFF")


class OutputControl2(OutputControl1):
    "Output 2"
    _prefix = "OO2"


class _PhaseValue(FloatUnitValue):
    _units = ['DG', 'RD']
    _name = "phase"


class PHA(_PhaseValue):
    "CHAN A PHA - Channel A phase"
class PHB(_PhaseValue):  # noqa
    "CHAN B PHA - Channel B phase"
class PHC(_PhaseValue):  # noqa
    "CHAN C PHA - Channel C phase"
class PHD(_PhaseValue):  # noqa
    "CHAN D PHA - Channel D phase"


class _Waveform(_PerOutputMode):
    "waveform"
    SI = ModeCommand('Sine', 'SINE')
    RA = ModeCommand('Ramp', 'RAMP')
    TR = ModeCommand('Triangle', 'TRG')
    SQ = ModeCommand('Square', 'SQR')
    NS = ModeCommand('Noise', 'NOIS')
    DC = ModeCommand('DC', 'DC')


class WaveformA(_Waveform):
    "CHAN A FORM - Channel A waveform"
    _prefix = "WFA"
class WaveformB(_Waveform):  # noqa
    "CHAN B FORM - Channel B waveform"
    _prefix = "WFB"
class WaveformC(_Waveform):  # noqa
    "CHAN C FORM - Channel C waveform"
    _prefix = "WFC"
class WaveformD(_Waveform):  # noqa
    "CHAN D FORM - Channel D waveform"
    _prefix = "WFD"


class _Destination(_PerOutputMode):
    "Destination"
    AM = ModeCommand('AM', 'AM')
    FM = ModeCommand('FM', 'FM')
    PM = ModeCommand('PM', 'PM')
    DS = ModeCommand('DS', 'DS')
    PU = ModeCommand('PU', 'PU')
    OC1 = ModeCommand('OC1', 'OC1')
    OC2 = ModeCommand('OC2', 'OC2')
    OF = ModeCommand('OF', 'OF')


class HP8904Device(AbstractGPIBDevice):
    _accepts = ["^(?P<model>HP *8904A).*$"]
    _idn = "ID"

    def __init__(self, idn, address, controller):
        super(HP8904Device, self).__init__(idn, address, controller)


deviceRegister.register_manager(HP8904Device)
