# -*- coding: utf-8 -*-
# This program is free software; you can redistribute it and/or modify it under
# the terms of the GNU General Public License as published by the Free Software
# Foundation; either version 2 of the License, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details
#
# You should have received a copy of the GNU General Public License along with
# this program; if not, write to the Free Software Foundation, Inc.,
# 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
"""Copyright (c) 2007-2018 David Douard (Paris, FRANCE).
https://bitbucket.org/dddouard/pygpibtoolkit -- mailto:david.douard@sdfa3.org

trace_decoder
=============

Module for decoding a trace generated by the HP3562A DSA, using the
GPIB command "DDBN" (Dump Data BiNary).

This file can be exectued as a python script. Use '-h' for more informations.

"""

import numpy
from pygpibtoolkit.gpib_utils import (
    format_datablock_header, decode_datablock_header, read_datablock_trace,
    decode_float)

from pygpibtoolkit.HP3562A.enum_types import *  # noqa


HEADER = [("DSPFCT", "Display function", EDSP, 'h', 2),
          ("NELTS", 'Number of elements', int, 'h', 2),
          ("DSPELTS", 'Displayed elements', int, 'h', 2),
          ("NAVG", 'Number of averages', int, 'h', 2),
          ("CHSEL", 'Channel selection', ECH, 'h', 2),
          ("OVST", 'Overflow status', EOVR, 'h', 2),
          ("OVLP", 'Overlap percentage', int, 'h', 2),
          ("DOM", 'Domain', EDOM, 'h', 2),
          ("VOLTMEAS", 'Volts peak/rms', EVLT, 'h', 2),
          ("AMPLUNT", 'Amplitude units', EAMP, 'h', 2),
          ("XUNT", 'X axis units', EXAXIS, 'h', 2),
          ("AMTHLBL", 'Auto math label', str, 's', 14),
          ("TRLBL", 'Trace label', str, 's', 22),
          ("EULB1", 'EU label 1', str, 's', 6),
          ("EULB2", 'EU label 2', str, 's', 6),
          ("VALTYPE", 'Float/Interger', bool, 'h', 2),
          ("RVALTYPE", 'Complex/Real', bool, 'h', 2),
          ("LIVE", 'Live/Recalled', bool, 'h', 2),
          ("MATHRESU", 'Math result', bool, 'h', 2),
          ("RVALINPUTTYPE", 'Real/Complex input', bool, 'h', 2),
          ("LOGDATA", 'Log/Linear data', bool, 'h', 2),
          ("AMTH", 'Auto math', bool, 'h', 2),
          ("RTST", 'Real time status', bool, 'h', 2),
          ("EMEAS", 'Measurement mode', EMEAS, 'h', 2),
          ("", 'Window', EWIN, 'h', 2),
          ("", 'Demod type channel 1', EDEMOD, 'h', 2),
          ("", 'Demod type channel 2', EDEMOD, 'h', 2),
          ("", 'Demod active channel 1', bool, 'h', 2),
          ("", 'Demod active channel 2', bool, 'h', 2),
          ("", 'Average status', EAVG, 'h', 2),
          ("", 'Not used', int, 'hh', 4),
          ("", 'Samp freq/2 (real)', decode_float, None, 4),
          ("", 'Samp freq/2 (imag)', decode_float, None, 4),
          ("", 'Not used', decode_float, None, 4),
          ("", 'Delta X-axis', decode_float, None, 4),
          ("", 'Max range', decode_float, None, 4),
          ("", 'Start time value', decode_float, None, 4),
          ("", 'Expon wind const 1', decode_float, None, 4),
          ("", 'Expon wind const 2', decode_float, None, 4),
          ("", 'EU value chan 1', decode_float, None, 4),
          ("", 'EU value chan 2', decode_float, None, 4),
          ("", 'Trig delay chan 1', decode_float, None, 4),
          ("", 'Trig delay chan 2', decode_float, None, 4),
          ("", 'Start freq value', decode_float, None, 8),
          ("", 'Start data value', decode_float, None, 8),
          ]


def decode_trace(data, idx=0):
    """
    Decode the data (as generated by the HP3562A DSA in response to a
    "DDBN" command), and returns a couple (header, values).

    header is the dictionnary of the header of the dumped data block,
    value is a numpy array holding the trace (vector of float or
    complex values).
    """
    header, idx = decode_datablock_header(data, HEADER, idx)
    return header, read_datablock_trace(
        data, idx, header["Number of elements"])


def main():
    import sys
    import argparse
    opt = argparse.ArgumentParser("A simple tool for tracing a dumped trace")
    opt.add_argument(
        '-f', '--filename', default=None, dest='filename',
        help='Output filename. If not set, read from stdin')
    opt.add_argument(
        '-m', '--mode', default='binary',dest='mode',
        help='Dumping mode (may be "binary" [default], "ascii" or "ansi")')
    opt.add_argument(
        '-d', '--display-header', default=False, action="store_true",
        dest="displayheader", help="Display the trace header")
    opt.add_argument(
        '-P', '--noplot-trace', default=True, action="store_false",
        dest="plot", help="Do not display the plot of the trace")
    opt.add_argument(
        '-x', '--xmode', default='lin', dest='xmode',
        help='X coordinate mode (may be "lin" [default] or "log")')
    opt.add_argument(
        '-y', '--ymode', default='lin', dest='ymode',
        help='Y coordinate mode (may be "lin" [default], "log" or "db")')

    options = opt.parse_args()

    if options.filename is None:
        print("Can't deal stdin for now...")
        sys.exit(1)
    try:
        with open(options.filename, 'rb') as f:
            header, data = decode_trace(f.read())
    except Exception as e:
        print("ERROR: can't read %s an interpret it as a HP3562 trace"
              % options.filename)
        print(e)
        sys.exit(1)

    if options.displayheader:
        print(format_datablock_header(header, HEADER, 100))

    f0 = header['Start freq value']
    dx = header['Delta X-axis']
    n = header['Number of elements']
    x = numpy.linspace(f0, f0+dx*n, len(data))
    y = data.copy()
    if options.plot:
        import pylab
        if options.ymode != "lin":
            minv = min(y[y > 0])
            y[y == 0] = minv
            y = numpy.log10(y)
        if options.ymode == "db":
            y = y * 10
            pylab.ylabel('db')
        pylab.grid()
        pylab.plot(x, y)
        pylab.xlabel('frequency')
        pylab.show()
    return header, x, y


if __name__ == "__main__":
    h, x, y = main()
