#
# This program is free software; you can redistribute it and/or modify it under
# the terms of the GNU General Public License as published by the Free Software
# Foundation; either version 2 of the License, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details
#
# You should have received a copy of the GNU General Public License along with
# this program; if not, write to the Free Software Foundation, Inc.,
# 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
""" Copyright (c) 2007-2018 David Douard (Paris, FRANCE).
https://bitbucket.org/dddouard/pygpibtoolkit -- mailto:david.douard@sdfa3.org
"""

import os
import sys

from PyQt5 import QtWidgets, QtGui, QtCore
from PyQt5.QtCore import pyqtSignal

from pygpibtoolkit.tools import str_num_key

from pygpibtoolkit.qt5.qpreferences import (
    BaseItem, IntItem, UnicodeItem, ColorItem,
    PointItem, SizeItem, ByteArrayItem, AbstractPreferences)
from pygpibtoolkit.qt5.qpreferenceseditor import PreferencesEditor
import pygpibtoolkit.qt5.resources_rc
import pygpibtoolkit.qt5.icons_ressource_rc

# WARNING, may be "replaced" by mock
from pygpibtoolkit.HP3562A.dump_datablock import HP3562dumper

from pygpibtoolkit.HP3562A.datablockwidget import getChild


class Preferences(AbstractPreferences):
    ORGANISATION = "Logilab"
    APPLICATION = "q3562"

    _pos = PointItem()
    _size = SizeItem()
    _appState = ByteArrayItem()

    device = UnicodeItem(default='/dev/ttyUSB0',
                         name=u'device',
                         description=u'GPIB device',
                         group="GPIB settings")
    address = IntItem(default=0, min=0, max=16,
                      name=u'GPIB address',
                      group="GPIB settings")

    background = ColorItem(default=QtGui.QColor("white"),
                           name="Plots background",
                           group="Colors")


class Qt3562(QtWidgets.QMainWindow):
    def __init__(self, parent=None):
        super().__init__(parent)
        self.workspace = QtWidgets.QMdiArea(self)
        self.setCentralWidget(self.workspace)
        self.workspace.subWindowActivated.connect(self.updateMenus)
        self.windowMapper = QtCore.QSignalMapper(self)
        self.windowMapper.mapped.connect(self.workspace.setActiveSubWindow)

        self._receiving = False

        self.createActions()
        self.createMenus()
        self.createToolBars()
        self.createStatusBar()
        self.updateMenus()

        self._prefs = Preferences()
        self.initializeGPIB()
        if self._prefs._pos:
            self.move(self._prefs._pos)
        if self._prefs._size:
            self.resize(self._prefs._size)
        if self._prefs._appState:
            self.restoreState(self._prefs._appState)
        self.readPreferences()

    def readPreferences(self):
        bg = self._prefs.background
        # pen_colors = [self._prefs["color%d"%i] for i in range(8)]
        # self.plotterWidget.pen_colors = pen_colors

    def createActions(self):
        self.openAct = QtWidgets.QAction(
            QtGui.QIcon(":/icons/open.png"),
            self.tr("&Open..."), self,
            shortcut=self.tr("Ctrl+O"),
            statusTip=self.tr("Open an existing data block file"),
            triggered=self.open)

        self.saveAsAct = QtWidgets.QAction(
            self.tr("Save &As..."),
            self, statusTip=self.tr("Save the document under a new name"),
            triggered=self.saveAs)

        self.exitAct = QtWidgets.QAction(
            self.tr("E&xit"), self,
            shortcut=self.tr("Ctrl+Q"),
            statusTip=self.tr("Exit the application"),
            triggered=self.close)

        self.closeAct = QtWidgets.QAction(
            self.tr("Cl&ose"), self,
            shortcut=self.tr("Ctrl+W"),
            statusTip=self.tr("Close the active window"),
            triggered=self.workspace.closeActiveSubWindow)

        self.closeAllAct = QtWidgets.QAction(
            self.tr("Close &All"), self,
            statusTip=self.tr("Close all the windows"),
            triggered=self.workspace.closeAllSubWindows)

        self.tileAct = QtWidgets.QAction(
            self.tr("&Tile"), self,
            statusTip=self.tr("Tile the windows"),
            triggered=self.workspace.tileSubWindows)

        self.cascadeAct = QtWidgets.QAction(
            self.tr("&Cascade"), self,
            statusTip=self.tr("Cascade the windows"),
            triggered=self.workspace.cascadeSubWindows)

        #self.arrangeAct = QtWidgets.QAction(
        #    self.tr("Arrange &icons"), self,
        #    statusTip=self.tr("Arrange the icons"),
        #    triggered=self.workspace.arrangeIcons)

        self.nextAct = QtWidgets.QAction(
            self.tr("Ne&xt"), self,
            shortcut=self.tr("Ctrl+F6"),
            statusTip=self.tr("Move the focus to the next window"),
            triggered=self.workspace.activateNextSubWindow)

        self.previousAct = QtWidgets.QAction(
            self.tr("Pre&vious"), self,
            shortcut=self.tr("Ctrl+Shift+F6"),
            statusTip=self.tr("Move the focus to the previous window"),
            triggered=self.workspace.activatePreviousSubWindow)

        self.separatorAct = QtWidgets.QAction(self)
        self.separatorAct.setSeparator(True)

        self.aboutAct = QtWidgets.QAction(
            self.tr("&About"), self,
            statusTip=self.tr("Show the application's About box"),
            triggered=self.about)

        self.aboutQtAct = QtWidgets.QAction(
            self.tr("About &Qt"), self,
            statusTip=self.tr("Show the Qt library's About box"),
            triggered=QtWidgets.qApp.aboutQt)

        self.getStateAct = QtWidgets.QAction(
            QtGui.QIcon(":/icons/state.svg"),
            self.tr("Get state"), self,
            checkable=True,
            statusTip=self.tr("Retrieve State from GPIB device"),
            triggered=self.getState)

        self.getTraceAct = QtWidgets.QAction(
            QtGui.QIcon(":/icons/trace.svg"),
            self.tr("Get trace"), self,
            checkable=True,
            statusTip=self.tr("Retrieve Trace from GPIB device"),
            triggered=self.getTrace)

        self.getCoordAct = QtWidgets.QAction(
            QtGui.QIcon(":/icons/coord.svg"),
            self.tr("Get coord"), self,
            checkable=True,
            statusTip=self.tr("Retrieve Coord from GPIB device"),
            triggered=self.getCoord)

        self.channelActGroups = QtWidgets.QActionGroup(self, exclusive=True)
        self.channelAAct = QtWidgets.QAction(
            QtGui.QIcon(":/icons/displayA.svg"),
            self.tr("Channel A"), self,
            checkable=True, checked=True,
            statusTip=self.tr("Retrieve from channel A"))
        self.channelActGroups.addAction(self.channelAAct)

        self.channelBAct = QtWidgets.QAction(
            QtGui.QIcon(":/icons/displayB.svg"),
            self.tr("Channel B"), self,
            checkable=True, checked=False,
            statusTip=self.tr("Retrieve from channel B"))
        self.channelActGroups.addAction(self.channelBAct)

    def setupUi(self):
        self.actionPreferences.triggered.connect(self.preferencesTriggered)
        self.actionQuit.triggered.connect(self.quitTriggered)
        self.actionQuit.setShortcut(QtGui.QKeySequence(u'Ctrl+Q'))
        self.actionOpen.triggered.connect(self.openTriggered)
        self.actionOpen.setShortcut(QtGui.QKeySequence(u'Ctrl+O'))
        self.actionSave.triggered.connect(self.saveTriggered)
        self.actionSave.setShortcut(QtGui.QKeySequence(u'Ctrl+S'))
        self.actionSaveAs.triggered.connect(self.saveAsTriggered)

    def about(self):
        QtWidgets.QMessageBox.about(
            self, self.tr("About Q3562"),
            self.tr("A simple application to talk to the HP3562A DSA."))

    def updateMenus(self):
        hasMdiChild = (self.activeMdiChild() is not None)
        self.saveAsAct.setEnabled(hasMdiChild)
        self.closeAct.setEnabled(hasMdiChild)
        self.closeAllAct.setEnabled(hasMdiChild)
        self.tileAct.setEnabled(hasMdiChild)
        self.cascadeAct.setEnabled(hasMdiChild)
        # self.arrangeAct.setEnabled(hasMdiChild)
        self.nextAct.setEnabled(hasMdiChild)
        self.previousAct.setEnabled(hasMdiChild)
        self.separatorAct.setVisible(hasMdiChild)

    def updateWindowMenu(self):
        self.windowMenu.clear()
        self.windowMenu.addAction(self.closeAct)
        self.windowMenu.addAction(self.closeAllAct)
        self.windowMenu.addSeparator()
        self.windowMenu.addAction(self.tileAct)
        self.windowMenu.addAction(self.cascadeAct)
        # self.windowMenu.addAction(self.arrangeAct)
        self.windowMenu.addSeparator()
        self.windowMenu.addAction(self.nextAct)
        self.windowMenu.addAction(self.previousAct)
        self.windowMenu.addAction(self.separatorAct)

        windows = self.workspace.subWindowList()
        self.separatorAct.setVisible(len(windows) != 0)

        i = 0
        for child in windows:
            if i < 9:
                text = self.tr("&{0} {1}").format(
                    i + 1, child.userFriendlyName())
            else:
                text = self.tr("%{0} {1}").format(
                    i + 1, child.userFriendlyName())
            i += 1
            action = self.windowMenu.addAction(text)
            action.setCheckable(True)
            action.setChecked(child == self.activeMdiChild())
            action.triggered.connect(self.windowMapper.map)
            self.windowMapper.setMapping(action, child)

    def createMenus(self):
        self.fileMenu = self.menuBar().addMenu(self.tr("&File"))
        self.fileMenu.addAction(self.openAct)
        self.fileMenu.addAction(self.saveAsAct)
        self.fileMenu.addSeparator()
        self.fileMenu.addAction(self.exitAct)

        self.deviceMenu = self.menuBar().addMenu(self.tr("&Device"))
        self.deviceMenu.addAction(self.channelAAct)
        self.deviceMenu.addAction(self.channelBAct)
        self.deviceMenu.addSeparator()
        self.deviceMenu.addAction(self.getStateAct)
        self.deviceMenu.addAction(self.getTraceAct)
        self.deviceMenu.addAction(self.getCoordAct)

        self.windowMenu = self.menuBar().addMenu(self.tr("&Window"))
        self.windowMenu.aboutToShow.connect(self.updateWindowMenu)

        self.menuBar().addSeparator()
        self.helpMenu = self.menuBar().addMenu(self.tr("&Help"))
        self.helpMenu.addAction(self.aboutAct)
        self.helpMenu.addAction(self.aboutQtAct)

    def createToolBars(self):
        self.fileToolBar = self.addToolBar(self.tr("File"))
        self.fileToolBar.setObjectName('filetoolbar')
        self.fileToolBar.addAction(self.openAct)
        self.fileToolBar.addAction(self.saveAsAct)
        self.fileToolBar.addSeparator()

        self.deviceToolBar = self.addToolBar(self.tr("Device"))
        self.deviceToolBar.setObjectName('devicetoolbar')
        self.deviceToolBar.addAction(self.channelAAct)
        self.deviceToolBar.addAction(self.channelBAct)
        self.deviceToolBar.addSeparator()
        self.deviceToolBar.addAction(self.getStateAct)
        self.deviceToolBar.addAction(self.getTraceAct)
        self.deviceToolBar.addAction(self.getCoordAct)

    def createStatusBar(self):
        self.statusBar().showMessage(self.tr("Ready"))

    def preferencesTriggered(self, checked=False):
        PreferencesEditor(self._prefs, self).exec_()
        self.readPreferences()
        self.replotCurrent()

    def closeEvent(self, event):
        if 1:
        #if self.promptForSave():
            self._prefs._pos = self.pos()
            self._prefs._size = self.size()
            self._prefs._appState = self.saveState()
            event.accept()
        else:
            event.ignore()

    def open(self, checked=False):
        filenames = QtWidgets.QFileDialog.getOpenFileNames(
            self, "Open a dumped data block file", '.',
            'bin files (*.bin)\nAll files (*)')
        self.openFiles(filenames[0])

    def openFiles(self, filenames):
        ok = False
        for filename in filenames:
            filename = str(filename)
            if os.path.exists(filename):
                with open(filename, 'rb') as f:
                    data = f.read()
                name = os.path.basename(filename)
                # name = os.path.splitext(name)[0]
                child_cls = getChild(data)
                assert child_cls
                child = child_cls(data, name=name)
                self.workspace.addSubWindow(child)
                child.show()
                ok = True
        self.statusBar().showMessage(self.tr("File(s) loaded"), 2000)
        return ok

    def saveAs(self, checked=False):
        return
#         index = self.plotsView.selectionModel().currentIndex()
#         if index.isValid():
#             filename = QtGui.QFileDialog.getSaveFileName(self, "Selecte a file name to save HPGL file", '.', 'HPGL files (*.plt)\nAll files (*)')
#             n = index.row()
#             value = unicode(self.plotsView.model().data(index, Qt.DisplayRole).toString())
#             open(filename, 'w').write(self._plots[value])

    def activeMdiChild(self):
        return self.workspace.activeSubWindow()

    def initializeGPIB(self):
        try:
            self.gpib_dumper = HP3562dumper(device=self._prefs.device,
                                            address=self._prefs.address,
                                            )
            self.captureThread = GPIBConnection(self.gpib_dumper)
            self.captureThread.datablockDumped.connect(self.datablockReceived)
            self.captureThread.start()
        except Exception as e:
            print("ERROR: cannot initialize GPIB")
            print(e)
            self.gpib_plotter = None

    def datablockReceived(self, datablock):
        # datablock = datablock.data()
        child_cls = getChild(datablock)
        assert child_cls
        child = child_cls(datablock)
        self.workspace.addSubWindow(child)
        child.show()
        self.getStateAct.setChecked(False)
        self.getTraceAct.setChecked(False)
        self.getCoordAct.setChecked(False)
        self.statusBar().showMessage(self.tr("Received data block"), 2000)
        self._receiving = False
        # give control back to fron panel
        self.captureThread.sendCommand('LCL')

    def selectChannel(self):
        if self.channelAAct.isEnabled():
            self.captureThread.sendCommand("A")
        else:
            self.captureThread.sendCommand("B")

    def getState(self):
        if self._receiving:
            return
        self._receiving = True
        self.statusBar().showMessage(
            self.tr("Waiting for a %s data block") % (self.tr('state')))
        self.getStateAct.setChecked(True)
        if self.captureThread:
            self.captureThread.startCapture(mode="state")

    def getTrace(self):
        if self._receiving:
            return
        self._receiving = True
        self.statusBar().showMessage(
            self.tr("Waiting for a %s data block") % (self.tr('trace')))
        self.getTraceAct.setChecked(True)
        self.selectChannel()
        if self.captureThread:
            self.captureThread.startCapture(mode="trace")

    def getCoord(self):
        if self._receiving:
            return
        self._receiving = True
        self.statusBar().showMessage(
            self.tr("Waiting for a %s data block") % (self.tr('coord')))
        self.getCoordAct.setChecked(True)
        self.selectChannel()
        if self.captureThread:
            self.captureThread.startCapture(mode="coord")


class GPIBConnection(QtCore.QThread):
    datablockDumped = pyqtSignal(object)

    def __init__(self, cnx):
        QtCore.QThread.__init__(self)
        self.gpibdumper = cnx

        self._cancelmutex = QtCore.QMutex()
        self._cancel = False
        self._modemutex = QtCore.QMutex()
        self._mode = None
        # self._nreceived = 0
        self._startstopmutex = QtCore.QMutex()
        self._startstop = QtCore.QWaitCondition()
        self._capturing = False

    def cancel(self):
        self._cancelmutex.lock()
        self._cancel = True
        self._cancelmutex.unlock()

    def sendCommand(self, cmd):
        if self._capturing:
            return
        return self.gpibdumper.send_command(cmd)

    def startCapture(self, mode):
        self._modemutex.lock()
        self._mode = mode
        self._modemutex.unlock()
        self._startstop.wakeOne()

    def stopCapture(self):
        self._startstopmutex.lock()
        self._capturing = False
        self._startstopmutex.unlock()

    def run(self):
        while 1:
            self._cancelmutex.lock()
            if self._cancel:
                return
            self._cancelmutex.unlock()
            self._startstopmutex.lock()
            if not self._capturing:
                self._startstop.wait(self._startstopmutex)
                self._capturing = True
            self._startstopmutex.unlock()
            self._modemutex.lock()
            mode = self._mode
            self._modemutex.unlock()
            datablock = self.gpibdumper.dump(mode=mode)
            self._capturing = False
            if datablock:
                self.datablockDumped.emit(datablock)
            self.msleep(10)


def main():
    import optparse
    opt = optparse.OptionParser(
        'A simple GPIB-based nterface for the HP3562A DSA')
    opt.add_option('-m', '--mockup', default=False,
                   action="store_true",
                   dest='mockup',
                   help='Use a pseudo GPIB connection (for test purpose)',
                   )
    opt.add_option('-v', '--verbose', default=False,
                   action="store_true",
                   dest="verbose",
                   help="Verbose mode",)

    options, argv = opt.parse_args(sys.argv)

    if options.mockup:
        # overwrite "normal" module
        global HP3562dumper
        from pygpibtoolkit.HP3562A.dump_datablock_mockup import HP3562dumper

    a = QtWidgets.QApplication(argv)
    w = Qt3562()
    files = [f for f in argv[1:] if os.path.isfile(f)]
    files.sort(key=str_num_key)
    w.openFiles(files)

    w.show()
    a.exec_()

if __name__ == '__main__':
    main()
