# This program is free software; you can redistribute it and/or modify it under
# the terms of the GNU General Public License as published by the Free Software
# Foundation; either version 2 of the License, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details
#
# You should have received a copy of the GNU General Public License along with
# this program; if not, write to the Free Software Foundation, Inc.,
# 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
""" Copyright (c) 2007-2008 David Douard (Paris, FRANCE).
https://bitbucket.org/dddouard/pygpibtoolkit -- mailto:david.douard@sdfa3.org
"""

import os
from PyQt5 import QtWidgets, QtGui, uic
from PyQt5.QtCore import Qt
import sip

from pygpibtoolkit.qt5.qpreferences import UnicodeItem
from pygpibtoolkit.tools import AbstractRegister

form_class, base_class = uic.loadUiType(
    os.path.join(os.path.dirname(__file__), "qpreferences_dialog.ui"),
    import_from=True, from_imports='pygpibtoolkit.qt5')


class WidgetMetaclass(sip.wrappertype):
    _widgets = {}

    def __init__(cls, name, bases, dct):
        # called at class creation
        super().__init__(name, bases, dct)
        if name != "BaseWidget":
            WidgetRegister().add(cls)


class BaseWidget(QtWidgets.QWidget, metaclass=WidgetMetaclass):
    _filter = None


class WidgetRegister(AbstractRegister):
    _registered_type = BaseWidget
    getWidget = AbstractRegister.get_class


class ItemValidator(QtGui.QValidator):
    def __init__(self, parent, item):
        super().__init__(parent)
        self._item = item

    def validate(self, value, pos):
        value = str(value)
        if not value.strip():
            return (self.Intermediate, value, pos)
        if self._item.validate(value):
            return (self.Acceptable, value, pos)
        return (self.Invalid, value, pos)


class BaseEditor(BaseWidget):
    """
    Basic editor for preference items. Use a QLineEdit with no
    validation or so...
    """
    _accepts = "UnicodeItem"

    def __init__(self, parent, item):
        BaseWidget.__init__(self, parent)
        self._item = item
        self.setupUI()

    def setValue(self, value):
        self._editor.setText(str(value))

    def getValue(self):
        return str(self._editor.text())

    def setupUI(self):
        self._editor = QtWidgets.QLineEdit(self)
        self._validator = ItemValidator(self, self._item)
        self._editor.setValidator(self._validator)
        l = QtWidgets.QHBoxLayout(self)
        l.setContentsMargins(0, 0, 0, 0)
        l.addWidget(self._editor, 1)
        self.setFocusProxy(self._editor)


class IntEditor(BaseEditor):
    _accepts = "IntItem"

    def setupUI(self):
        self._editor = QtWidgets.QSpinBox(self)
        self._editor.setMinimum(self._item._min)
        self._editor.setMaximum(self._item._max)
        l = QtWidgets.QHBoxLayout(self)
        l.setContentsMargins(0,0,0,0)
        l.addWidget(self._editor, 1)
        self.setFocusProxy(self._editor)

    def setValue(self, value):
        self._editor.setValue(int(value))

    def getValue(self):
        return self._editor.value()


class BoolEditor(BaseEditor):
    _accepts = "BoolItem"

    def setupUI(self):
        self._editor = QtWidgets.QCheckBox(self)
        l = QtWidgets.QHBoxLayout(self)
        l.setContentsMargins(0,0,0,0)
        l.addWidget(self._editor, 1)
        self.setFocusProxy(self._editor)

    def setValue(self, value):
        if value:
            value = Qt.Checked
        else:
            value = Qt.Unchecked
        self._editor.setCheckState(value)

    def getValue(self):
        return self._editor.checkState() == Qt.Checked


class ColorEditor(BaseEditor):
    _accepts = "ColorItem"

    def setupUI(self):
        self._editor_pix = QtGui.QPixmap(40,30)
        self._editor_pix.fill(QtGui.QColor('white'))

        self._editor_btn = QtWidgets.QPushButton("")
        self._editor_btn.setFlat(True)
        self._editor_btn.setFocusPolicy(Qt.NoFocus)
        self._editor_btn.setIcon(QtGui.QIcon(self._editor_pix))

        self._editor_edt = QtWidgets.QLineEdit()
        self._editor_edt.setInputMask(r"\#HHHHHHhh")
        fm = QtWidgets.QApplication.fontMetrics()
        w = fm.width("#FFFFFFFF   ")
        self._editor_edt.setMaximumWidth(w)

        l = QtWidgets.QHBoxLayout(self)
        l.setContentsMargins(0,0,0,0)
        l.addWidget(self._editor_edt)
        l.addWidget(self._editor_btn)
        l.addStretch(1)
        self.setFocusProxy(self._editor_edt)
        self._editor_btn.pressed.connect(self.chooseColor)
        self._editor_edt.editingFinished.connect(self.colorEdited)

    def setValue(self, value):
        if isinstance(value, tuple):
            color = self._item._type(*value)
        elif isinstance(value, self._item._type):
            color = value
        elif isinstance(value, long):
            color = self._item._type(value)
            alpha = value >> 24
            color.setAlpha(alpha)
        else:
            color = self._item._type(value)

        rgba = color.getRgb()
        colorname = ("#"+"%02X"*4)%rgba
        self._rgba = rgba
        self._editor_pix.fill(color)
        self._editor_btn.setIcon(QtGui.QIcon(self._editor_pix))
        self._editor_edt.setText(colorname)

    def getValue(self):
        return self._item._type(*self._rgba)

    def colorEdited(self):
        val = unicode(self._editor_edt.text())
        if len(val) == 7:
            val += "FF" # miss alpha channel
        val = val[1:]
        val = [val[2*i:2*i+2] for i in range(len(val)/2)]
        val = [int(x, 16) for x in val]

        self._rgba = tuple(val)
        self.setValue(self._rgba)

    def chooseColor(self):
        newcolor, ok = QtWidgets.QColorDialog.getRgba(self.getValue().rgba(), self)
        if ok:
            self.setValue(newcolor)


class PreferencesEditor(QtWidgets.QDialog, form_class):
    def __init__(self, preferences, parent=None):
        super().__init__(parent)
        self.setupUi(self)
        self._prefs = preferences
        self.buildUI()

    def buildUI(self):
        mainw = self.centralTab
        for i in range(mainw.count()):
            mainw.removeTab(0)

        eds = {}
        self._editors = eds

        wr = WidgetRegister()
        if len(self._prefs.groups)>1:
            for group in self._prefs.groups:
                if group is None:
                    continue
                w = QtWidgets.QWidget(mainw)
                mainw.addTab(w, group)
                g = QtWidgets.QGridLayout(w)
                g.setVerticalSpacing(2)
                for i, k in enumerate(self._prefs.keys(group)):
                    name = self._prefs.getName(k)
                    item = self._prefs.getItem(k)
                    if not name:
                        name = k
                    l = QtWidgets.QLabel(name, w)
                    g.addWidget(l, i, 0)
                    if self._prefs.getDescription(k):
                        l.setToolTip(self._prefs.getDescription(k))
                    wcls = wr.getWidget(item)
                    e = wcls(w, item)
                    eds[k] = e
                    g.addWidget(e, i, 1)
                    val = self._prefs.getPref(k)
                    if val is None:
                        val = ''
                    e.setValue(val)

                # add blank space
                g.addWidget(QtWidgets.QWidget(w), i+1, 0)
                g.setRowStretch(i+1, 1)
                g.setColumnStretch(1, 1)

    def accept(self):
        p = self._prefs
        for k in self._editors:
            newval = self._editors[k].getValue()
            p.setPref(k, newval)
        return super().accept()


if __name__ == '__main__':
    from pygpibtoolkit.qt5.qpreferences import (
        AbstractPreferences, UnicodeItem, IntItem, BaseItem)
    from pygpibtoolkit.qt5.qpreferences import ColorItem

    class TestPreferences(AbstractPreferences):
        ORGANISATION = "PyGPIToolkit"
        APPLICATION = "test_qpref_editor"

        device = UnicodeItem('/dev/ttyUSB0', name="the device",
                             group="GPIB settings")
        address = IntItem(5, name="address",
                          description="GPIB address of the plotter",
                          group="GPIB settings",
                          min=0, max=16)
        other = UnicodeItem('toto', name="other stuff",
                             group="General")
        color = ColorItem(default='red',name="Colour",
                          group="General")
        _pos = BaseItem(None)
        _size = BaseItem(None)
        _appState = BaseItem(None)

    a = QtWidgets.QApplication([])

    prefs = TestPreferences()
    w = PreferencesEditor(prefs)
    w.show()
    a.exec_()
