# This program is free software; you can redistribute it and/or modify it under
# the terms of the GNU General Public License as published by the Free Software
# Foundation; either version 2 of the License, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details
#
# You should have received a copy of the GNU General Public License along with
# this program; if not, write to the Free Software Foundation, Inc.,
# 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
""" Copyright (c) 2007-2018 David Douard (Paris, FRANCE).
https://bitbucket.org/dddouard/pygpibtoolkit -- mailto:david.douard@sdfa3.org

"""

from pygpibtoolkit.pygpib import Constants, Command
from pygpibtoolkit.pygpib import Condition
from pygpibtoolkit.pygpib import BoolValue, IntValue
from pygpibtoolkit.pygpib import PercentageValue, FrequencyValue, DurationValue
from pygpibtoolkit.pygpib import VoltageValue
from pygpibtoolkit.pygpib import EnumValue, StringValue
from pygpibtoolkit.pygpib import Mode, ModeCommand, Flag

from pygpibtoolkit.gpibcontroller import AbstractGPIBDevice, deviceRegister


#####################
# HP3562A constants and command set
# **VERY INCOMPLETE**

class LinRes(Condition):
    def __call__(self, device):
        return device.MeasMode == "LINEAR RES"
class LogRes(Condition):  # noqa
    def __call__(self, device):
        return device.MeasMode == "LOG RES"
class SweptSine(Condition):  # noqa
    def __call__(self, device):
        return device.MeasMode == "SWEPT SINE"
class TimeCaptur(Condition):  # noqa
    def __call__(self, device):
        return device.MeasMode == "TIME CAPTUR"


# GPIB buffer size is 3x80 characters lines
class STATUS_BYTE(Constants):
    # HP3562A Status Byte, as returned by a serial poll
    _constants = [(0x40, "RQS", "Request Service"),
                  # when sent in response to a serial poll
                  (0x20, "ERR", "GPIB error"),
                  (0x10, "RDY", "ready to accept GPIB commands"),
                  ]

    conditions = [(0, "NSR", "No service requested"),
                  (1, "USRQ1", "User SRQ #1"),
                  (2, "USRQ1", "User SRQ #2"),
                  (3, "USRQ1", "User SRQ #3"),
                  (4, "USRQ1", "User SRQ #4"),
                  (5, "USRQ1", "User SRQ #5"),
                  (6, "USRQ1", "User SRQ #6"),
                  (7, "USRQ1", "User SRQ #7"),
                  (8, "USRQ1", "User SRQ #8"),
                  (9, "EOD", "End of disk action"),
                  (10, "EOP", "End of plot action"),
                  (11, "STCH", "Instrument status changed"),
                  # STCH: any change in the status register sets this condition
                  (12, "PWR", "Power on"),
                  (13, "KEY", "Key pressed"),
                  (14, "DCP", "Device clear plotter (listen)"),
                  # ...
                  ]

    def __init__(self):
        super(STATUS_BYTE, self).__init__()
        self._conditions = dict([(x[0], x[1]) for x in self.conditions])
        self._rev_conditions = dict([(x[1], x[0]) for x in self.conditions])
        self._long_conditions = dict([(x[0], x[2]) for x in self.conditions])

    def byte_condition(self, byte):
        byte = byte & 0x8F
        return self._conditions.get(byte, "N/A")


class IS(Flag):
    "INTERNAL STATUS - Internal Status Register"
    _constants = [(0x01, "MEASP", "measeurement pause"),
                  (0x02, "ASQP", "Auto sequence pause"),
                  (0X04, "EOM", "End of measurement, capture or throughput"),
                  (0x08, "EOAS", "End of auto sequence"),
                  (0x10, "SWPR", "Sweep point ready"),
                  (0x20, "CH1OV", "Channel 1 overrange"),
                  (0x40, "CH2OV", "Channel 2 overrange"),
                  (0X80, "CH1HR", "Channel 1 half range"),
                  (0x100, "CH2HR", "Channel 2 half range"),
                  (0x200, "SFALT", "Source fault"),
                  (0x400, "RUNL", "Reference unlock"),
                  (0x800, "RMKT", "Remote marker knob turn"),
                  (0x1000, "REKT", "Remote entry knob turn"),
                  (0x2000, "ASRC", "Asctive Status Register changed"),
                  (0x4000, "PWRF", "Power-on test failed"),
                  ]


class STA(Flag):
    "STATUS QUERY - Status/event query"
    _constants = [(0x01, "N/A", "Not used"),
                  (0x02, "N/A", "Not used"),
                  (0x04, "KEY", "Key pressed"),
                  (0x08, "N/A", "Not used"),
                  (0x10, "RDY", "Ready"),
                  (0x20, "ERR", "Error"),
                  (0x40, "RQS", "Request"),
                  (0x80, "MOS", "Message on screen"),
                  (0x100, "MEASP", "Measurement pause"),
                  (0x200, "ASQP", "Auto sequence pause"),
                  (0x400, "EOM", "End of measurement, capture or throughput"),
                  (0x800, "EOAS", "End of auto sequence"),
                  (0x1000, "SWPR", "Sweep point ready"),
                  (0x2000, "CH1OV", "Channel 1 overrange"),
                  (0x4000, "CH2OV", "Channel 2 overrange"),
                  (0x8000, "MAOV", "Math overflow"),
                  ]


class AS(Flag):
    "ACTIVITY STATUS - Activity Status Register"
    _constants = [(0x01, "CKFL", "Check fault log"),
                  (0x02, "FITR", "Filling time record"),
                  (0x04, "FLTR", "Filters settings"),
                  (0x08, "CFTP", "Curve fir in progress"),
                  (0x10, "MSSM", "Missed sample"),
                  (0x20, "TMPR", "Timed preview"),
                  (0x40, "ACDA", "Accept data"),
                  # ...
                  ]


class TraceDisplay(Mode):
    "Trace Display"
    A = ModeCommand("A trace", "A")
    B = ModeCommand("B trace", "B")
    AB = ModeCommand("A & B traces", "A&B")

    def get_mode(self, device):
        # XXX Can I get this from the device?
        # mode = device.DSBN. # HERE
        return None


class ARM(Command):
    "ARM - Arm"


class NAVG(IntValue):
    "Number of averages"


class AverageMode(Mode):
    "Average mode"
    _condition = LinRes() | LogRes() | TimeCaptur()
    _key = "EAVGTYP"
    _DATA_BLOCK = "DSBN"

    STBL = ModeCommand(6, "STABLE (MEAN)")
    EXP = ModeCommand(7, "EXPON")
    PHLD = ModeCommand(8, "PEAK HOLD")
    CNPK = ModeCommand(9, "CONT PEAK")
    AVOF = ModeCommand(10, "AVG OFF")


class TIAV(BoolValue):
    "TIM AV ON OFF - Time average"
    _condition = LinRes() | TimeCaptur() | SweptSine()


class OVLP(PercentageValue):
    "OVRLP% - Overlap (%)"
    _condition = LinRes() | TimeCaptur() | SweptSine()


class AUTO(BoolValue):
    "AUTO ON OFF - Auto calibration"


class SNGC(Command):
    "SINGLE CAL - Single calibration"


class CoordMode(Mode):
    "Coord mode"
    _key = "EYCOORD"
    _DATA_BLOCK = "DCBN"

    MGDB = ModeCommand(5, "MAG (dB)")
    MDBM = ModeCommand(10, "MAG (dBm)")
    MGLG = ModeCommand(4, "MAG (LOG)")
    MAG = ModeCommand(3, "MAG (LIN)")
    PHSE = ModeCommand(8, "PHASE")
    REAL = ModeCommand(1, "REAL")
    IMAG = ModeCommand(2, "IMAG")
    NYQT = ModeCommand(6, "NYQUIST")
    NICL = ModeCommand(9, "NICHOL")


# FREQ menu
class FRS(FrequencyValue):
    "FREQ SPAN - Freq. span"


class SF(FrequencyValue):
    "START FREQ - Start freq."


class CF(FrequencyValue):
    "CENTER FREQ - Center freq."
    _units = "hz", "khz", "mhz", "ord", "rmp"
    _condition = LinRes() | TimeCaptur() | SweptSine()


class ZST(Command):
    "ZERO START - Zero start"
    _condition = LinRes() | TimeCaptur() | SweptSine()


class MAXS(FrequencyValue):
    "MAX SPAN - Max span"
    _condition = LinRes() | TimeCaptur() | SweptSine()


class TLN(DurationValue):
    "TIME LENGTH - Time len."
    _condition = LinRes() | TimeCaptur() | SweptSine()


class ESMP(BoolValue):
    "E SMPL ON OFF - E sample"
    _condition = LinRes() | TimeCaptur() | SweptSine()


class SMPF(FrequencyValue):
    "SAMPLE FREQ - Sample freq."
    _condition = LinRes() | TimeCaptur() | SweptSine()


class SPF(FrequencyValue):
    "STOP FREQ - Stop freq."
    _condition = SweptSine


class SWRT(FrequencyValue):
    "SWEEP RATE - Sweep rate"


class ABIB(Command):
    "ABORT HP-IB - Abort HPIB"


# INPUT COUPLING
class C1AC(EnumValue):
    "CHAN1 AC DC - Channel 1 AC/DC"
    _values = ["DC", "AC"]


class Chan1Coupling(Mode):
    "Channel 1 coupling"
    _key = "FLT1"
    _DATA_BLOCK = "DSBN"
    FLT1 = ModeCommand(1, "FLOAT CHAN1")
    GND1 = ModeCommand(0, "GROUND CHAN1")


class C2AC(EnumValue):
    "CHAN2 AC DC - Channel 2 AC/DC"
    _values = ["AC", "DC"]


class Chan2Coupling(Mode):
    "Channel 2 coupling"
    _key = "FLT1"
    _DATA_BLOCK = "DSBN"
    FLT2 = ModeCommand("Float", "FLOAT CHAN2")
    GND2 = ModeCommand("Ground", "GROUND CHAN2")


class LCL(Command):
    "LOCAL - Local"


class C1RG(VoltageValue):
    "CHAN 1 RANGE - Channel 1 range"
    _units = "V", "mV", "Vrms", "mVrms", "dBV", "EU"


class C2RG(VoltageValue):
    "CHAN 2 RANGE - Channel 2 range"
    _units = "V", "mV", "Vrms", "mVrms", "dBV", "EU"


# MEAS MODE
class MeasMode(Mode):
    "Measurement mode"
    _key = "EMEAS"
    _DATA_BLOCK = "DSBN"
    LNRS = ModeCommand(0, "LINEAR RES")
    LGRS = ModeCommand(1, "LOG RES")
    SSIN = ModeCommand(2, "SWEPT SINE")
    CPTR = ModeCommand(3, "TIME CAPTUR")


# MEAS DISP
class FreqResp(Condition):
    def __call__(self, device):
        return device.SelectMeas == "FREQ RESP"


class MeasDisp(Mode):
    # TODO
    FRQR = ModeCommand(0, "FREQ RESP",
                       condition=((LinRes() | LogRes())
                                  & FreqResp()
                                  | SweptSine()))
    COHR = ModeCommand(0, "COHER",
                       condition=((LinRes() | LogRes())
                                  & FreqResp()
                                  | SweptSine()))
    PSP1 = ModeCommand(0, "POWER SPEC1",
                       condition=((LinRes() | LogRes())
                                  & FreqResp()
                                  | SweptSine()))
    PSP2 = ModeCommand(0, "POWER SPEC2",
                       condition=((LinRes() | LogRes())
                                  & FreqResp()
                                  | SweptSine()))
    CSPS = ModeCommand(0, "CROSS SPEC",
                       condition=((LinRes() | LogRes())
                                  & FreqResp()
                                  | SweptSine()))
    IRSP = ModeCommand(0, "IMPLS RESP",
                       condition=LinRes() & FreqResp())
    AUMT = ModeCommand(0, "AUTO MATH", )
    FILT = ModeCommand(0, "FILTRD INPUT")

    TMR1 = ModeCommand(0, "TIME REC1", condition=TimeCaptur)
    TMR2 = ModeCommand(0, "TIME REC2", condition=TimeCaptur)
    LSP1 = ModeCommand(0, "LINEAR SPEC1", condition=TimeCaptur)
    LSP2 = ModeCommand(0, "LINEAR SPEC2", condition=TimeCaptur)


class PowerSpec(Condition):
    def __call__(self, device):
        return device.SelectMeas == "POWER SPEC"


class AutoCorr(Condition):
    def __call__(self, device):
        return device.SelectMeas == "AUTO CORR"


class CrossCorr(Condition):
    def __call__(self, device):
        return device.SelectMeas == "CROSS CORR"


class Hist(Condition):
    def __call__(self, device):
        return device.SelectMeas == "HIST"


# SELECT MEAS
class SelectMeas(Mode):
    "Select measurement"
    _key = "EMTYP"
    _DATA_BLOCK = "DSBN"
    _condition = LinRes() | TimeCaptur() | SweptSine()

    FRSP = ModeCommand(
        0, "FREQ RESP", condition=LinRes() | LogRes() | SweptSine())
    PSPC = ModeCommand(
        2, "POWER SPEC", condition=LinRes() | LogRes() | TimeCaptur())
    AUCR = ModeCommand(
        3, "AUTO CORR", condition=LinRes() | TimeCaptur())
    CCOR = ModeCommand(
        1, "CROSS CORR", condition=LinRes())
    HIST = ModeCommand(
        4, "HIST", condition=LinRes() | TimeCaptur())


class Channel(Mode):
    ""
    _key = "ECH"
    _DATA_BLOCK = "DDBN"  # XXX sure?
    CH12 = ModeCommand("Channel 1&2 active", "CH 1&2 ACTIVE",
                       condition=LinRes() | LogRes())
    CH1 = ModeCommand("Channel 1 active", "CH 1 ACTIVE",
                      condition=LinRes() | LogRes() | TimeCaptur())
    CH2 = ModeCommand("Channel 2 active", "CH 2 ACTIVE",
                      condition=LinRes() | LogRes() | TimeCaptur())


class REV(StringValue):
    _readonly = True
    "REVISION - Revision"


class SACR(Command):
    "SACR - Send Auto Carrier"


class HP356XDevice(AbstractGPIBDevice):
    _accepts = [r"^(?P<model>HP *356[23][ABCD]).*$", ]
    _idn = "ID?"

    def manage_srq(self, statusbyte):
        print("Managing SRQ: %s" % statusbyte)


deviceRegister.register_manager(HP356XDevice)
