# -*- coding: utf-8 -*-
# This program is free software; you can redistribute it and/or modify it under
# the terms of the GNU General Public License as published by the Free Software
# Foundation; either version 2 of the License, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details
#
# You should have received a copy of the GNU General Public License along with
# this program; if not, write to the Free Software Foundation, Inc.,
# 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
""" Copyright (c) 2007-2018 David Douard (Paris, FRANCE).
https://bitbucket.org/dddouard/pygpibtoolkit -- mailto:david.douard@sdfa3.org

"""
import mx.DateTime as dt

from pygpibtoolkit.pygpib import Command
from pygpibtoolkit.pygpib import ModeCommand
from pygpibtoolkit.pygpib import WriteOnlyMode
from pygpibtoolkit.pygpib import BoolValue, FloatValue
from pygpibtoolkit.gpibcontroller import AbstractGPIBDevice, deviceRegister


class Function(WriteOnlyMode):
    "Function"
    _init_value = "S0F1"

    # S0 = ModeCommand("Unshift", "S0")
    # S1 = ModeCommand("Shift", "S1")

    S0F1 = ModeCommand("DC Voltage", "DCV")
    S0F2 = ModeCommand("AC Voltage", "ACV")
    S0F3 = ModeCommand("AC + DC Voltage", "ACV+DCV")
    S0F4 = ModeCommand("2 wire Ohm", "OHM2")
    S0F5 = ModeCommand("4 wire Ohm", "OHM4")

    S1F1 = ModeCommand("DC/DC Voltage Ratio", "DCV/DCV")
    S1F2 = ModeCommand("AC/DC Voltage Ratio", "ACV/DCV")
    S1F3 = ModeCommand("(AC + DC)/DC Voltage Ratio", "ACV+DCV/DCV")
    S1F4 = ModeCommand("O.C. 2 wire Ohm", "OCO2")
    S1F5 = ModeCommand("O.C. 4 wire Ohm", "OCO4")


class Range(WriteOnlyMode):
    "Range"
    _init_value = "R1"

    R1 = ModeCommand("Auto", "AUTO")
    R2 = ModeCommand("0.1", "0.1")
    R3 = ModeCommand("1", "1")
    R4 = ModeCommand("10", "10")
    R5 = ModeCommand("100", "100")
    R6 = ModeCommand("1E3", "1E3")
    R7 = ModeCommand("10E3", "10E3")
    R8 = ModeCommand("100E3", "100E3")
    R9 = ModeCommand("1E6", "1E6")


class Trigger(WriteOnlyMode):
    "Trigger"
    _init_value = "T1"

    T1 = ModeCommand("Internat", "INT")
    T2 = ModeCommand("External", "EXT")
    T3 = ModeCommand("Single", "SGL")
    T4 = ModeCommand("Hold", "HLD")


class Math(WriteOnlyMode):
    "Math"
    _init_value = "M0"

    M0 = ModeCommand("Off", "OFF")
    M1 = ModeCommand("Pass/Fail", "P/F")
    M2 = ModeCommand("Statistics", "STAT")
    M3 = ModeCommand("Null", "NULL")
    M4 = ModeCommand("dBm", "DBM")
    M5 = ModeCommand(u"Thermistor (°F)", "TH_F")
    M6 = ModeCommand(u"Thermistor (°C)", "TH_C")
    M7 = ModeCommand("Scale", "")
    M8 = ModeCommand("% Error", "")
    M9 = ModeCommand("dB", "")


class AutoZero(WriteOnlyMode):
    "Auto Zero"
    _init_value = "Z1"

    Z0 = ModeCommand("Off", "OFF")
    Z1 = ModeCommand("On", "ON")


class Filter(WriteOnlyMode):
    "Filter"
    _init_value = "FL0"

    FL0 = ModeCommand("Off", "OFF")
    FL1 = ModeCommand("On", "ON")


class Test(WriteOnlyMode):
    "Test"
    _init_value = "TE0"

    TE0 = ModeCommand("Off", "OFF")
    TE1 = ModeCommand("On", "ON")


class ReadingStorage(WriteOnlyMode):
    "Reading Storage"
    _init_value = "RS0"

    RS0 = ModeCommand("Off", "OFF")
    RS1 = ModeCommand("On", "ON")


class SystemOutputMode(WriteOnlyMode):
    "System Output Mode"
    _init_value = "SO0"

    SO0 = ModeCommand("Off", "OFF")
    SO1 = ModeCommand("On", "ON")


class Display(WriteOnlyMode):
    "Display"
    _init_value = "D1"

    D0 = ModeCommand("Off", "OFF")
    D1 = ModeCommand("On", "ON")


class OutputFormat(WriteOnlyMode):
    "Output Format"
    _init_value = "P0"

    P0 = ModeCommand("ASCII", "ASC")
    P1 = ModeCommand("Packed Format", "PFOR")


class EOI(WriteOnlyMode):
    "EOI"
    _init_value = "O1"

    O0 = ModeCommand("Disable", "OFF")
    O1 = ModeCommand("Enable", "ON")


class SW1(BoolValue):
    _readonly = True

    def convert_from(self, *value):
        if value:
            return int(value[0]) and "Front" or "Rear"
        return "?"


class H(Command):
    "HOME - Software Reset"


class CL1(Command):
    "CLEAR CONTINUE - Active"


class _Register(FloatValue):
    _readonly = False
    _cached = False

    def build_get_cmd(self):
        return "RE" + self.__class__.__name__

    def build_set_cmd(self, *value):
        value = self.convert_to(*value)
        cmd = "%sST%s" % (value, self.__class__.__name__)
        return cmd, value


class N(_Register):
    "Number of readings"


class G(_Register):
    "Number of displayed digits"


class I(_Register):
    "Number of power line cycles int."


class D(_Register):
    "Delay"


class M(_Register):
    "Mean"
    _readonly = True


class V(_Register):
    "Variance"
    _readonly = True


class C(_Register):
    "Count"
    _readonly = True


class L(_Register):
    "Lower"


class R(_Register):
    "R"


class U(_Register):
    "Upper"


class Y(_Register):
    "Y"


class Z(_Register):
    "Z"


class HP3456Device(AbstractGPIBDevice):
    _accepts = [r"^(?P<model>HP3456A)$", ]
    _idn = None  # cannot ask device its model number or version

    def __init__(self, idn, address, controller):
        super(HP3456Device, self).__init__(idn, address, controller)
        self.reset_measures()
        self._data_cb = None

    def manage_srq(self, statusbyte):
        # print "Managing SRQ", statusbyte
        if statusbyte & 0x04:
            # data ready, read the value on the GPIB bus
            self._controller.send_command(self._address, '', cb=self.data_cb)

    def data_cb(self, data):
        if data:
            try:
                data = float(data)
                self.measures.append((dt.now(), data))
                if self._data_cb:
                    self._data_cb(self.measures[-1])
            except:
                pass

    def reset_measures(self):
        self.measures = []

    def register_data_cb(self, cb):
        self._data_cb = cb


deviceRegister.register_manager(HP3456Device)
