# This program is free software; you can redistribute it and/or modify it under
# the terms of the GNU General Public License as published by the Free Software
# Foundation; either version 2 of the License, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details
#
# You should have received a copy of the GNU General Public License along with
# this program; if not, write to the Free Software Foundation, Inc.,
# 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
""" Copyright (c) 2007-2018 David Douard (Paris, FRANCE).
https://bitbucket.org/dddouard/pygpibtoolkit -- mailto:david.douard@qsda3.org
"""

from PyQt5 import QtCore, QtGui

from pygpibtoolkit.tools import AbstractRegister


class PreferenceMetaclass(type):
    _widgets = {}

    def __init__(cls, name, bases, dct):
        # called at class creation
        super().__init__(name, bases, dct)
        if name != "BaseItem":
            ItemRegister().add(cls)


class BaseItem:
    # __metaclass__ = PreferenceMetaclass
    _id = 0

    def __init__(self, default=None, name=None, description=None, group=None):
        self._default = default
        self._id = BaseItem._id
        self._name = name
        self._description = description
        self._group = group
        BaseItem._id += 1

    def validate(self, value):
        return True

    def __get__(self, obj, cls):
        if obj is None:  # when called from the class, return the Item itself
            return self
        try:
            return self._type(obj.getPref(self._id))
        except Exception as e:
            return None

    def __set__(self, obj, value):
        obj.setPref(self._id, value)


class ItemRegister(AbstractRegister):
    _registered_type = BaseItem
    getItem = AbstractRegister.get_class


class PointItem(BaseItem):
    _type = QtCore.QPoint


class SizeItem(BaseItem):
    _type = QtCore.QSize


class ByteArrayItem(BaseItem):
    _type = QtCore.QByteArray


class UnicodeItem(BaseItem):
    _type = str

    def validate(self, value):
        return isinstance(value, str)


class IntItem(BaseItem):
    _type = int

    def __init__(self, default=None, name=None, description=None,
                 group=None, min=None, max=None):
        super().__init__(default, name, description, group)
        self._min = min
        self._max = max

    def validate(self, value):
        try:
            value = self._type(value)
        except:
            return False
        if self._min is not None and value < self._min:
            return False
        if self._max is not None and value > self._max:
            return False
        return True


class ColorItem(BaseItem):
    _type = QtGui.QColor

    def validate(self, value):
        try:
            self._type(value)
            return True
        except:
            return False


class BoolItem(BaseItem):
    _type = bool

    def validate(self, value):
        try:
            self._type(value)
            return True
        except:
            return False


class AbstractPreferences(QtCore.QObject):
    def __init__(self):
        super().__init__()
        self._settings = QtCore.QSettings(QtCore.QSettings.UserScope,
                                          self.ORGANISATION, self.APPLICATION)
        self._prefs = {}
        self.groups = []
        keys = []
        for k in dir(self.__class__):
            item = self.getItem(k)
            if isinstance(item, BaseItem):
                keys.append((k, item))
        keys.sort(key=lambda x: x[1]._id)
        for k, item in keys:
            self._prefs[item._id] = k
            if item._group not in self.groups:
                self.groups.append(item._group)

    def getItem(self, key):
        return getattr(self.__class__, key)
        # return self._prefs.get(key, None)

    def getPref(self, key):
        key = self._prefs.get(key, key)
        default = getattr(self.__class__, key)._default
        val = self._settings.value(key, default)
        return val

    def setPref(self, key, value):
        key = self._prefs.get(key, key)
        self._settings.setValue(key, QtCore.QVariant(value))

    def keys(self, group=None):
        return [k for k in self._prefs.values()
                if not k.startswith('_') and self.getItem(k)._group == group]

    def getName(self, key):
        item = getattr(self.__class__, key)
        return item._name

    def getDescription(self, key):
        item = getattr(self.__class__, key)
        return item._description

    def __getitem__(self, key):
        return getattr(self, key)
