# This program is free software; you can redistribute it and/or modify it under
# the terms of the GNU General Public License as published by the Free Software
# Foundation; either version 2 of the License, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details
#
# You should have received a copy of the GNU General Public License along with
# this program; if not, write to the Free Software Foundation, Inc.,
# 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
""" Copyright (c) 2007-2008 David Douard (Paris, FRANCE).
https://bitbucket.org/dddouard/pygpibtoolkit -- mailto:david.douard@sdfa3.org

Module defining a mockup GPIB communication object.

"""

class BaseMockup(object):
    _idnstr = "BaseMockup"
    _idncmd = "*IDN?"

    _cmds = {}
    
    def __init__(self):
        #self._current_cmd = None
        self._current_data = None
        
    def cmd(self, cmd, *args):
        if cmd == self._idncmd:
            return self._idnstr
        try:
            if ' ' in cmd:
                cmd, cmdargs = cmd.split(None, 1)
                args = args + (cmdargs,)
            _cmd = getattr(self, cmd)
            self._current_data = _cmd(*args)
        except AttributeError:
            if cmd in self._cmds:
                self._current_data = iter(self._cmds[cmd])
            else:
                raise ValueError('Unknown command %s'%repr(cmd))
        else:
            if isinstance(self._current_data, basestring):
                ret = self._current_data
                self._current_data = None
                return ret
            elif self._current_data is None:
                return None
            else:
                try:
                    return self._current_data.next()
                except StopIteration:
                    self._current_data = None
                    return None        
        return None

    def read(self):
        try:
            return self._current_data.next()
        except StopIteration:
            self._current_data = None
            return None        
        except:
            return None
    
GPIB_CONTROLLER = 1
GPIB_DEVICE = 0

class GPIB(object):
    _retries = 15
    def __init__(self, address=0, mode=1, **kw):
        """
        Create a new mockup GPIB controller.        
        """
        self.set_mode(mode)
        self.set_address(address)
        self._devices = {}

    def adddevice(self, add, mockup):
        self._devices[add] = mockup
            
    def set_address(self, address, check=True):
        """
        Set the address of the GPIB device:
        
        - if the device is the Controller In Charge, this is the
          address of the device commands are sent to,

        - if the device is in GPIB_DEVICE mode, this is its address.
        """
        self._address = address
        
    def set_mode(self, mode):
        """
        Set GPIB device mode to 'mode':
        
        - GPIB_CONTROLLER: set the device as the Controller In Charge
          on the GPIB bus

        - GPIB_DEVICE: set the device as a standard GPIB device on the
          bus.
        """
        self._mode = mode

    def set_controller(self):
        """
        Set GPIB device the Controller In Charge on the GPIB bus.
        """
        self.set_mode(1)

    def set_device(self):
        """
        Set the GPIB device as a simple device on the GPIB bus.        
        """
        self.set_mode(0)

    def send_command(self, cmd, address=None):
        """
        Send the specified GPIB command on the bus (must be the CIC),
        and read the answer.
        Eventually, set the addressed device first.
        """
        assert self._mode == 1
        if address is not None:
            self.set_address(address)

        if self._address in self._devices:
            return self._devices[self._address].cmd(cmd)
        return None
        
    def read_eoi(self, address=None):
        """
        Read the HPIB buffer from device, till EOI is performed, or timeout.
        """
        if address is not None:
            self.set_address(address)
        if self._address in self._devices:
            return "".join(self._devices[self._address].read())
        ret = ""
        return ret
            
    def check_srq(self):
        """
        Check the SRQ line
        """
        assert self._mode == 1, "must be the Controller In Charge"
        return None

    def poll(self, addresses=None):
        """
        Poll every address, and return a dictionnary
         {add: status, ...}        
        """        
        assert self._mode == 1, "must be the Controller In Charge"
        ret = {}
        if addresses is None:
            for add, mock in self._devices.items():
                ret[add] = mock._idnstr
            return ret
        return {}
    
    def reset(self):
        """
        Perform a reset of the USB device
        
        """
        pass
        
