# -*- coding: utf-8 -*-
# This program is free software; you can redistribute it and/or modify it under
# the terms of the GNU General Public License as published by the Free Software
# Foundation; either version 2 of the License, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details
#
# You should have received a copy of the GNU General Public License along with
# this program; if not, write to the Free Software Foundation, Inc.,
# 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
"""Copyright (c) 2007-2018 David Douard (Paris, FRANCE).
https://bitbucket.org/dddouard/pygpibtoolkit -- mailto:david.douard@sdfa3.org

state_decoder
=============

Module for decoding the internal state of the HP3562A DSA, using the
GPIB command "DSBN" (Dump State BiNary).

This file can be exectued as a python script. Use '-h' for more informations.

"""

from pygpibtoolkit.gpib_utils import (
    decode_datablock_header, read_datablock_trace,
    decode_float, format_datablock_header)
from pygpibtoolkit.HP3562A.trace_decoder import HEADER as TRACE_HEADER
from pygpibtoolkit.HP3562A.enum_types import *  # noqa


HEADER = [("EYCOORD", "Y coordinates", EYCOORD, 'h', 2),
          ("", "# of disp elements", int, 'h', 2),
          ("", "First element", int, 'h', 2),
          ("", "Total elements", int, 'h', 2),
          ("", "Display sampling", EDISPSMP, 'h', 2),
          ("", "Scaling", ESCAL, 'h', 2),
          ("", "Data pointer", int, 'l', 4),
          ("", "In data", int, 'l', 4),
          ("", "Log/linear x-axis", bool, 'h', 2),
          ("", "Sampled display data", bool, 'h', 2),
          ("", "Plot/Graph mode", bool, 'h', 2),
          ("", "Phase wrap", bool, 'h', 2),
          ("", "Not used", None, None, 36),
          ("", "X scale factor", decode_float, None, 4),
          ("", "Grid min Y scale", decode_float, None, 4),
          ("", "Grid max Y scale", decode_float, None, 4),
          ("", "/div", decode_float, None, 4),
          ("", "Min value of data", decode_float, None, 4),
          ("", "Max value of data", decode_float, None, 4),
          ("", "Y cumulative min", decode_float, None, 4),
          ("", "Y cumulative max", decode_float, None, 4),
          ("", "Y scale factor", decode_float, None, 4),
          ("", "Not used", None, None, 16),
          ("", "Stop value", decode_float, None, 8),
          ("", "Left grid", decode_float, None, 8),
          ("", "Right grid", decode_float, None, 8),
          ("", "Left data", decode_float, None, 8),
          ("", "Right data", decode_float, None, 8),
          ]


def decode_coord(data):
    """
    Decode the data (as generated by the HP3562A DSA in response to a
    "DSBN" command), and returns a dict (header)

    header is the dictionnary of the header of the dumped data block,
    """
    header, idx = decode_datablock_header(data, HEADER)
    trace_header, idx = decode_datablock_header(data, TRACE_HEADER, idx)
    trace = read_datablock_trace(data, idx, trace_header["Number of elements"])
    return header, trace_header, trace


def main():
    import sys
    import argparse
    import numpy

    opt = argparse.ArgumentParser(
        "A simple tool for displaying dumped coord data block")
    opt.add_argument(
        '-f', '--filename', default=None, dest='filename',
        help='Input filename. If not set, read from stdin')
    opt.add_argument(
        '-m', '--mode', default='binary', dest='mode',
        help='Dumping mode (may be "binary" [default], "ascii" or "ansi")')
    opt.add_argument(
        '-d', '--display-header', default=False, action="store_true",
        dest="displayheader", help="Display the trace header")
    opt.add_argument(
        '-P', '--noplot-trace', default=True, action="store_false",
        dest="plot", help="Do not display the plot of the trace")
    opt.add_argument(
        '-x', '--xmode', default='lin', dest='xmode',
        help='X coordinate mode (may be "lin" [default] or "log")')
    opt.add_argument(
        '-y', '--ymode', default='lin', dest='ymode',
        help='Y coordinate mode (may be "lin" [default], "log" or "db")')

    options = opt.parse_args()

    if options.filename is None:
        print("Can't deal stdin for now...")
        sys.exit(1)
    try:
        with open(options.filename, 'rb') as f:
            coord_header, header, data = decode_coord(f.read())
    except Exception as e:
        print("ERROR: can't read %s or interpret it as a HP3562 trace"
              % options.filename)
        print(e)
        raise
        sys.exit(1)

    if options.displayheader:
        print(format_datablock_header(coord_header, HEADER, 100))
        print(format_datablock_header(header, TRACE_HEADER, 100))
    if options.plot:
        f0 = header['Start freq value']
        dx = header['Delta X-axis']
        n = header['Number of elements']
        x = numpy.linspace(f0, f0+dx*n, len(data))
        y = data.copy()

        ym = coord_header['Min value of data']
        yM = coord_header['Max value of data']
        ys = coord_header['Y scale factor']

        y[y < ym] = ym
        y *= ys

        import pylab
        pylab.ylabel(header['Amplitude units'])
        pylab.grid()
        pylab.plot(x, y)
        pylab.xlabel('frequency (%s)' % header["X axis units"])
        pylab.ylabel("%s (%s)" % (
            coord_header['Y coordinates'], header['Amplitude units']))
        pylab.show()


if __name__ == "__main__":
    main()
