# -*- coding: utf-8 -*-
# This program is free software; you can redistribute it and/or modify it under
# the terms of the GNU General Public License as published by the Free Software
# Foundation; either version 2 of the License, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details
#
# You should have received a copy of the GNU General Public License along with
# this program; if not, write to the Free Software Foundation, Inc.,
# 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
"""Copyright (c) 2007-2018 David Douard (Paris, FRANCE).
https://bitbucket.org/dddouard/pygpibtoolkit -- mailto:david.douard@sdfa3.org

state_decoder
=============

Module for decoding the internal state of the HP3562A DSA, using the
GPIB command "DSBN" (Dump State BiNary).

This file can be exectued as a python script. Use '-h' for more informations.

"""

from pygpibtoolkit.gpib_utils import (
    format_datablock_header, decode_datablock_header,
    decode_float, decode_string)

from pygpibtoolkit.HP3562A.enum_types import *  # noqa


HEADER = [("EMEAS", "Measurement mode", EMEAS, 'h', 2),
          ("MTYP1", "Measurement 1", EMTYP, 'h', 2),
          ("MTYP2", "Measurement 2", EMTYP, 'h', 2),
          ("WNDO", "Window type", EWINTYP, 'h', 2),
          ("FOXP1", "Force/Expon window 1", EFEXPW, 'h', 2),
          ("FOXP2", "Force/Expon window 2", EFEXPW, 'h', 2),
          ("EAVGTYP", "Average type", EAVGTYP, 'h', 2),
          ("OVLP", "Overlap percentage", int, 'h', 2),
          ("NAVG", "Number of averages", int, 'h', 2),
          ("SWAVG", "Sweep # of averages", int, 'h', 2),
          ("TRGTYP", "Trigger type", ETRGTYP, 'h', 2),
          ("TRGSLP", "Trigger slope", ETRGSLP, 'h', 2),
          ("PRVTYP", "Preview type", EPRVTYP, 'h', 2),
          ("SMPF", "Sample type", ESMPTYP, 'h', 2),
          ("RNG1", "Range units chan 1", ERNGUNT, 'h', 2),
          ("RNG2", "Range units chan 2", ERNGUNT, 'h', 2),
          ("RNGTYP1", "Range type 1", ERNGTYP, 'h', 2),
          ("RNGTYP2", "Range type 2", ERNGTYP, 'h', 2),
          ("CPL1", "Input coupling 1", EINCPL, 'h', 2),
          ("CPL2", "Input coupling 2", EINCPL, 'h', 2),
          ("SRCTYP", "Source type", ESRCTYP, 'h', 2),
          ("CHRP", "Chirp percent", int, 'h', 2),
          ("BRST", "Burst percent", int, 'h', 2),
          ("SWPDIR", "Sweep direction", ESWPDIR, 'h', 2),
          ("SWPMOD", "Sweep mode", ESWPMOD, 'h', 2),
          ("EFRQUNT", "Ext sample freq units", EXAXIS, 'h', 2),
          ("BWUNT", "Bandwidth units", EXAXIS, 'h', 2),
          ("LGSP", "Log span index", int, 'h', 2),
          ("LGID", "Log start index", int, 'h', 2),
          ("SWUNT", "Sweep rate units", EXAXIS, 'h', 2),
          ("AGREFCH", "Auto gain ref chan", EDEMODCH, 'h', 2),
          ("DMDCH", "Demod channels", EDEMODCH, 'h', 2),
          ("DMDTYP1", "Demod type chan 1", EDEMOD, 'h', 2),
          ("DMDTYP2", "Demod type chan 2", EDEMOD, 'h', 2),
          ("SRLVUNT", "Source level units", EXAXIS, 'h', 2),
          ("SROFUNT", "Source offset units", EXAXIS, 'h', 2),
          ("TRLVUNT", "Trigger level units", EXAXIS, 'h', 2),
          ("", "Capt/thru length units", EXAXIS, 'h', 2),
          ("EULB1", "EU label 1", str, 's', 6),
          ("EULB2", "EU label 2", str, 's', 6),
          ("", "Auto carrier on/off", bool, 'h', 2),
          ("", "Time average on/off", bool, 'h', 2),
          ("", "Auto/fixed resolution", bool, 'h', 2),
          ("", "Auto gain on/off", bool, 'h', 2),
          ("", "Auto/fixed integrate",bool, 'h', 2),
          ("", "Fast average on/off", bool, 'h', 2),
          ("", "Overload reject on/off", bool, 'h', 2),
          ("FLT1", "Chan 1 float/ground", bool, 'h', 2),
          ("FLT2", "Chan 2 float/ground", bool, 'h', 2),
          ("", "Time throughput on/off", bool, 'h', 2),
          ("", "Demodulation on/off", bool, 'h', 2),
          ("", "EU/volts chan 1", bool, 'h', 2),
          ("", "EU/volts chan 2", bool, 'h', 2),
          ("", "Manual/auto arm", bool, 'h', 2),
          ("", "Demod preview on/off", bool, 'h', 2),
          ("", "Delete freq on/off", bool, 'h', 2),
          ("", "Lin res Fstart pegged", bool, 'h', 2),
          ("", "Swept Fstart pegged", bool, 'h', 2),
          ("", "Force length chan 1", decode_float, None, 4),
          ("", "Force length chan 2", decode_float, None, 4),
          ("", "Expon time constant 1", decode_float, None, 4),
          ("", "Expon time constant 2", decode_float, None, 4),
          ("", "Sweep time", decode_float, None, 4),
          ("", "Sweep rate", decode_float, None, 4),
          ("", "Sweep resolution", decode_float, None, 4),
          ("", "Sweep integrate time", decode_float, None, 4),
          ("", "Auto gain level", decode_float, None, 4),
          ("", "Auto gain limit", decode_float, None, 4),
          ("", "Source level", decode_float, None, 4),
          ("", "EU value chan 1", decode_float, None, 4),
          ("", "EU value chan 2", decode_float, None, 4),
          ("", "Trigger delay chan 1", decode_float, None, 4),
          ("", "Trigger delay chan 2", decode_float, None, 4),
          ("", "Integrate var thresh", decode_float, None, 4),
          ("", "Capt/thru length", decode_float, None, 4),
          ("", "Frequency span", decode_float, None, 4),
          ("", "Time record length", decode_float, None, 4),
          ("", "Frequency resolution", decode_float, None, 4),
          ("", "Time resolution", decode_float, None, 4),
          ("", "External sample rate", decode_float, None, 4),
          ("", "Sample rate (actual)", decode_float, None, 4),
          ("", "Range channel 1", decode_float, None, 4),
          ("", "Range channel 2", decode_float, None, 4),
          ("", "Preview time", decode_float, None, 4),
          ("", "Trigger level", decode_float, None, 4),
          ("", "Source dc offset", decode_float, None, 4),
          ("", "Fixed sine frequency", decode_float, None, 8),
          ("", "Start frequency", decode_float, None, 8),
          ("", "Center frequency", decode_float, None, 8),
          ("", "Sweep start", decode_float, None, 8),
          ("", "Sweep end", decode_float, None, 8),
          ("", "Carrier frequency", decode_float, None, 8),
          ]


def decode_state(data):
    """
    Decode the data (as generated by the HP3562A DSA in response to a
    "DSBN" command), and returns a dict (header)

    header is the dictionnary of the header of the dumped data block,
    """
    header, idx = decode_datablock_header(data, HEADER)
    return header


def main():
    import sys
    import argparse
    opt = argparse.ArgumentParser("A simple tool for tracing a dumped trace")
    opt.add_argument(
        '-f', '--filename', default=None, dest='filename',
        help='Input filename. If not set, read from stdin')
    opt.add_argument(
        '-m', '--mode', default='binary', dest='mode',
        help='Dumping mode (may be "binary" [default], "ascii" or "ansi")')

    options = opt.parse_args()

    if options.filename is None:
        print("Can't deal stdin for now...")
        sys.exit(1)
    try:
        with open(options.filename, 'rb') as f:
            header = decode_state(f.read())
    except Exception as e:
        print("ERROR: can't read %s or interpret it as a HP3562 trace"
              % options.filename)
        print(e)
        raise
        sys.exit(1)

    print(format_datablock_header(header, HEADER, 100))


if __name__ == "__main__":
    main()
