# This program is free software; you can redistribute it and/or modify it under
# the terms of the GNU General Public License as published by the Free Software
# Foundation; either version 2 of the License, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details
#
# You should have received a copy of the GNU General Public License along with
# this program; if not, write to the Free Software Foundation, Inc.,
# 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
"""Copyright (c) 2007-2018 David Douard (Paris, FRANCE).
https://bitbucket.org/dddouard/pygpibtoolkit -- mailto:david.douard@sdfa3.org

"""

import sys
import time
import pygpibtoolkit.pygpib  # noqa
from pygpibtoolkit import prologix


class HP3562dumper(prologix.GPIB):
    """
    Class dedicated to dump data blocks from HP3562A DSA (trace,
    internal state or coordinate).
    """
    MODES = {
        'trace': 'DD',
        'state': 'DS',
        'coord': 'DC',
    }

    FORMATS = {
        'binary': 'BN',
        'ascii': 'AS',
        'ansi': 'AN',
    }

    def __init__(self, device="/dev/ttyUSB0", baudrate=115200, timeout=0.1,
                 address=0):
        super().__init__(device, baudrate, timeout, address, mode=1)

    def dump(self, mode='trace', format="binary"):
        """
        Dump the required data block and return it as a raw string buffer.

        'mode' can be 'trace' [default], 'state' or 'coord'.
        'format' can be 'binary' [default], 'ascii' or 'ansi'.
        """
        format = format.lower()
        mode = mode.lower()
        assert mode in self.MODES
        assert format in self.FORMATS
        cmd = self.MODES[mode] + self.FORMATS[format]

        res = ""
        print("command = %s" % cmd)
        # warning, this use direct low level serial communications...
        self._cnx.write('%s\r' % cmd)
        i = 0
        while i < self._retries:
            l = self._cnx.readline()
            if l.strip() == "":
                i += 1
                time.sleep(self._timeout)
                continue
            res += l
            i = 0
        return res


def main():
    import argparse
    opt = argparse.ArgumentParser(
        "A simple tool for dumping the current trace")
    opt.add_argument(
        '-f', '--filename', default=None, dest='filename',
        help='Output filename. If not set, write to stdout')
    opt.add_argument(
        '-d', '--device', default='/dev/ttyUSB0', dest='device',
        help='Device of the RS232 connection (default: /dev/ttyUSB0)')
    opt.add_argument(
        '-a', '--address', default=0, dest='address',
        help='GPIB address of the device')
    opt.add_argument(
        '-b', '--block', default='trace', dest='block',
        help=('Data block to dump (may be "trace" [default], '
              '"state" or "coord")'))
    opt.add_argument(
        '-m', '--mode', default='binary', dest='mode',
        help='Dumping mode (may be "binary" [default], "ascii" or "ansi")')

    options = opt.parse_args()

    cnx = HP3562dumper(device=options.device, address=int(options.address))
    res = cnx.dump(mode=options.block, format=options.mode)
    sys.stderr.write("read %s bytes\n" % (len(res)))
    if options.filename:
        open(options.filename, 'w').write(res)
    else:
        print(res)


if __name__ == '__main__':
    main()
