#include "stdio.h"
#include "mbed.h"
#include "string"
#include "Terminal6x8.h"
#include "Mono19x27.h"
#include "Mono15x22.h"
#include "Arial12x12.h"

#include "BufferedSerial.h"

#include "SSD1322.h"

Serial pc(USBTX, USBRX);
BufferedSerial hp(NC, PA_1, 64, 0); // serial4

SSD1322 dsp(SPI_8, 10000000, PB_15, PB_14, PB_13, PB_12, D11, D12, "SSD1322", 256, 64); 

#define MAX_BUFF 14
uint8_t curchar;
uint8_t cmd;
uint8_t nchars;
char buffer[MAX_BUFF+1];

typedef struct _DSP
{
  uint8_t cmd;
  uint8_t color;
  uint8_t bgcolor;
  uint8_t x0;
  uint8_t y0;
  uint8_t fmt; // 0=>ascii, 1=>hex, 2=>bits, 3=>flags
  uint8_t maxsize;
  uint8_t width;
  const unsigned char* font;
  char buffer[MAX_BUFF+1];
} DSP;

static DSP table[] =
{
  { 0x00, 0xF,   0x0,  0,  0, 0, MAX_BUFF, 245, Mono19x27}, // main display
  { 0x0C, 0xF,   0x0,204, 38, 0, 3,        45, Mono15x22}, // channels display
  { 0x01, 0xF,   0x0,  0,  0, 4, MAX_BUFF, 0, Terminal6x8}, 
  { 0x02, 0xF,   0x0,  0,  0, 4, MAX_BUFF, 0, Terminal6x8},
  { 0x0A, 0xF,   0x0,  0, 57, 2, 4,        0, Terminal6x8},
  { 0x0A, 0xF,   0x0,  0,  0, 3, 4,        0, Terminal6x8}, // flags
  { 0xFF, 0xF,   0x0,  0,  0, 4, MAX_BUFF, 0, Terminal6x8},
};

// 9x10
const unsigned char icon_alarm[] __attribute__((aligned (2))) = 
{
  0x1c, 0x0,
  0x3e, 0x0,
  0x7f, 0x0,
  0x7f, 0x0,
  0x7f, 0x0,
  0x7f, 0x0,
  0x7f, 0x0,
  0x7f, 0x0,
  0xff, 0x80,
  0x10, 0x0
};

const unsigned char icon_curve[] __attribute__((aligned (2))) = 
{
  0x80, 0x0,
  0x80, 0x0,
  0x80, 0x80,
  0x81, 0x0,
  0x9e, 0x0,
  0xa0, 0x0,
  0xc0, 0x0,
  0x80, 0x0,
  0x80, 0x0,
  0xff, 0x80
 };
 
typedef struct _FLAG
{
  uint8_t flag;
  uint8_t x;
  uint8_t y;
  const char* msg;
  const unsigned char* icon;
} FLAG;

static const FLAG flags[] =
  {
    //{ 0x00, 160, 30, "Alarm"},  // for the 'alarm' box
    { 0x00, 246, 0,  NULL, icon_alarm},      // F1.1
    { 0x01, 246, 11, NULL, icon_curve},      // F1.2
    { 0x03, 204, 30, "Channel"},    // F1.4
    { 0x14, 68,  44, "4W"},         // F2.5
    { 0x33, 40, 44,  "VIEW"},       // F4.4
    { 0x34, 0, 30,   "MON"},        // F4.5
    { 0x36, 0, 44,   "CONFIG"},     // F4.7
    
    { 0xFF, 0, 0,  "SCAN"},  // F4.6 or F4.3
    { 0xFF, 0, 0,  "*"},
    { 0xFF, 0, 0,  "ADRS"},
    { 0xFF, 0, 0,  "RMT"},
    { 0xFF, 0, 0,  "ERROR"},
    { 0xFF, 0, 0,  "EXT"},
    { 0xFF, 0, 0,  "ONCE"},
    { 0xFF, 0, 0,  "AUTO"},
    { 0xFF, 0, 0,  "MEM"},
    { 0xFF, 0, 0,  "LAST"},
    { 0xFF, 0, 0,  "MIN"},
    { 0xFF, 0, 0,  "MAX"},
    { 0xFF, 0, 0,  "AVG"},
    { 0xFF, 0, 0,  "OC"},

    { 0xFF, 0, 0,  "H"},
    { 0xFF, 0, 0,  "1"},
    { 0xFF, 0, 0,  "2"},
    { 0xFF, 0, 0,  "3"},
    { 0xFF, 0, 0,  "4"},
    { 0xFF, 0, 0,  "L"},
    
  };


#ifdef DEBUG
#define DBGPIN PC_0

DigitalOut dbgpin(DBGPIN);
inline void DebugPulse(uint8_t count=1)
{
  while (count--)
  {
    dbgpin = 1;
    dbgpin = 0;
  }
}
#else
inline void DebugPulse(uint8_t count=1)
{}
#endif

void test_dsp();

void setup() {
    // init the LCD

    //dsp.set_orientation(3);
    pc.baud (115200);
    pc.printf("\n\nSystem Core Clock = %.3f MHZ\r\n",
          (float)SystemCoreClock/1000000);

    hp.baud(187500);

    // myLCD.set_font((unsigned char*) Terminal6x8);
    // myLCD.claim(stdout);      // send stdout to the LCD display
    // myLCD.claim(stderr);      // send stderr to the LCD display
    dsp.background(Black);    // set background to black
    dsp.foreground(0xF);
    dsp.cls();
    dsp.locate(0, 0);
    dsp.set_font(Mono19x27);
    
    cmd = 0xFF;
    curchar = 0;
    nchars = 0;
    
    dsp.printf("HP34970A");
    dsp.set_font(Terminal6x8);
    for(uint8_t i=0; i<4;  i++) {
      dsp.locate(160, i*8);
      dsp.printf("Lg %d", i+1);
      dsp.locate(208, (3-i)*8);
      dsp.printf("Lg %d", i+1);
    }
    dsp.copy_to_lcd();
    wait(2);
    dsp.cls();
    test_dsp();
    wait(2);
    dsp.cls();
}

void go_msg() {
  dsp.set_font((unsigned char*) Arial12x12);
  dsp.locate(0, 38);
}

void show(uint8_t cmd, char *txt, uint8_t nchar=0) {
  uint8_t i, len;
  uint16_t bgcolor, fgcolor;
  char *oldv;

  len = MAX_BUFF;

  for (i=0; i<sizeof(table)/sizeof(table[0]); ++i) {
    if (table[i].cmd == cmd) {
      bgcolor = table[i].bgcolor;
      fgcolor = table[i].color;
      dsp.background(bgcolor);
      dsp.foreground(fgcolor);
      dsp.locate(table[i].x0, table[i].y0);
      dsp.set_font((unsigned char*) table[i].font);
      oldv = table[i].buffer;
      
      switch (table[i].fmt) {
      case 0: //ascii
	if (table[i].width > 0)
	  dsp.fillrect(table[i].x0, table[i].y0, table[i].x0 + table[i].width, table[i].y0 + table[i].font[2], bgcolor);
	dsp.printf(txt);
	break;
      case 1: // hex
	for (uint8_t j=0;; j++) {
	  if (txt[j] == 0x00)
	    break;
	  dsp.printf("%02X ", txt[j]);
	}
	for (uint8_t j=3*strlen(txt); j<table[i].maxsize; j++)
	  dsp.printf(" ");
	break;
      case 2: // binary
	dsp.foreground(fgcolor);
	dsp.printf(" [");
	for (uint8_t j=0; j<max(nchar, table[i].maxsize) ; j++) {
	  if (j>0) {
	    dsp.foreground(fgcolor);
	    dsp.printf(" | ");
	  }
	  for (uint8_t k=0; k<8; k++) {
	    if (txt[j] & (1 << (7-k)))
	      dsp.foreground(fgcolor);
	    else
	      dsp.foreground(bgcolor);
	    dsp.printf("%d", (8-k));
	  }
	}
	dsp.foreground(fgcolor);
	dsp.printf("]");
	break;
      case 3: // flags
	for (uint8_t j=0; j<max(nchar, table[i].maxsize) ; j++) {
	  for (uint8_t k=0; k<8; k++) {
	    if ((txt[j] & (1 << k) ) != (oldv[j] & (1 << k))) {
	 
	      if (txt[j] & (1 << k))
		dsp.foreground(fgcolor);
	      else
		dsp.foreground(bgcolor);
	      for (uint8_t l=0;
		   l<(sizeof(flags)/sizeof(flags[0])); ++l) {
		if (flags[l].flag == ((j<<4) + k)) {
		  if (flags[l].msg != NULL) { // a string
		    dsp.locate(flags[l].x, flags[l].y);
		    dsp.printf(flags[l].msg);}
		  else { // an icon
		    Bitmap_s pic = {9, 10, 2, flags[l].icon};
		    dsp.Bitmap_BW(pic, flags[l].x, flags[l].y);
		  }
		  break;
		}
	      }
	    }
	  }
	  oldv[j] = txt[j];
	}
	break;
      case 4: //ignore
	break;
      }
    }
  }
  dsp.copy_to_lcd();

}

void test_dsp()
{
  show(0x00, "8g8g8g8g8g8g8", 13); // main dsp
  show(0x0C, "888", 3); // channel dsp
  show(0x0A, "\xFF\xFF\xFF\xFF", 4); // flags
  dsp.copy_to_lcd();
}


void loop() { // run over and over
  if (hp.readable()) {
      uint8_t val = hp.getc();
      DebugPulse(3);
      // 0xFF: idle
      // 0xFE: frame finished
      // 0x66: Start of transmission received
      pc.printf("  VAL=0x%X\tCMD=0x%X\n", val, cmd);
      if (cmd == 0xFF) {
          if ((val == 0x66) || (val == 0x99)) {
              cmd = val;
          } else if ((val == 0x00) || (val == 0x55)){
              // probably an ACK for a keypad related event
          } else {
              // display "junk" byte
              //DebugPulse();
              show(0xFF, "", 0);
	      dsp.printf("%02X:> %02x", cmd, val);
          }

      } else if (cmd == 0xFE ) {
          if ((val == 0x66) || (val == 0x99)) {
              cmd = val;
          } else if (val == 0x55) {
              cmd = 0xFF;
          } else if (val == 0x00){
              // probably an ACK for a keypad related event
          } else {
              // display "junk" byte
              DebugPulse();
              show(0xFF, "", 5);
              dsp.printf("%02X=> %02x", cmd, val);
          }

      } else if (cmd == 0x66) { // waiting for the display command
	if ((val == 0x0C) || (val == 0x00) || (val == 0x01) || (val == 0x02) || (val == 0x0A)) {
              cmd = val;
              nchars = 0;
          } else {
              cmd = 0xFE;
              // display unknown cmd byte
              DebugPulse();
              show(0xFF, "", 10);
              dsp.printf("%02X-> %02x", cmd, val);
          }

      } else if (cmd == 0x99) { // waiting for a 0x00, it's the DP that sent a keypress event
        if (val != 0x00) {
              show(0xFF, "", 0);
	      dsp.printf("%02X kp %02X", cmd, val);
        }
        cmd = 0xFF;

      } else if (nchars == 0) { // waiting for the number of chars to display
          if (val>MAX_BUFF) {
              // display warning
              //dsp();
              //dsp << cmd << " got len " << val;
              //DebugPulse();
              show(0xFF, "", 0);
	      dsp.printf("%02X len %d", cmd, val);
              cmd = 0xFE; // weird
          } else {
              nchars = val;
              curchar = 0;
          }

      } else { // a character to display
          buffer[curchar] = val;
          curchar++;
          if (curchar == nchars) {
              buffer[curchar] = 0x00;
              show(cmd, buffer, nchars);
              nchars = 0;
              curchar = 0;
              cmd = 0xFE;
          }
      }

      DebugPulse(4);
      
  }
}

int main()
{
  setup();
  while(1)
    loop();
}
