#include "stdio.h"

#include <mbed.h>
#include <rtos.h>
#include <string>

#include "hp34comm.h"
#include "display.h"

#include "QEI.h"
#include "Keypad.h"

// Pins and device declarations
#if defined TARGET_NUCLEO_F446RE
#include "def_f446re.h"
#elif defined TARGET_HP34970_FP_F303RD
#include "def_hp34970_fp.h"
#endif

#include "platform/CircularBuffer.h"


Display *dsp;
volatile bool splashscreen;
HPSerial *hp;
Thread tdsp, tloop;
Ticker blinker;
Timeout rst_delay;
Timeout splashscreen_timer;
InterruptIn rst(HP_RST);

QEI qenc(KP_ENC1, KP_ENC2, NC, 16);
volatile uint8_t knob;
bool shift;  // true when kp is shifted, cleared by command 0x01 from Unit

typedef enum {
		KEY_NONE=0,
		KEY_PRESSED,
		KEY_RELEASED
} key_event_t;

typedef struct keycode {
	uint8_t row;
	uint8_t col;
	key_event_t keyevent;
} keycode_t;

//volatile keycode_t cur_keycode;

#define KEY_BUF_SIZE 10
CircularBuffer<keycode_t, KEY_BUF_SIZE> key_buf;

#define KP_NROWS 4
#define KP_NCOLS 5
DigitalIn kp_rows[KP_NROWS] = {
		KP_R0,
		KP_R1,
		KP_R2,
		KP_R3
};

DigitalOut kp_columns[KP_NCOLS] = {
		KP_C0, KP_C1,
		KP_C2, KP_C3,
		KP_C4
};

/* key mapping
   RxC   code   name
	 0x2   0x00   View
	 0x1   0x01   Mon
	 3x3   0x02   Sto/Rcl
	 0x0   0x03   Scan
	 1x2   0x04   Alarm
	 1x1   0x05   Mx+B
	 1x0   0x06   Measure
	 2x0   0x07   Interval
	 3x2   0x08   Card Reset
	 3x1   0x09   Close
	 3x0   0x0A   Open
	 0x3   0x0B   Read
	 2x3   0x0C   Shift
	 1x3   0x0D   Write
	 0x4   0x0E   Left
	 1x4   0x0F   Right
	 2x2   0x10   Advanced
	 2x1   0x11   Step
*/

uint8_t kp_mapping[KP_NROWS][KP_NCOLS] = {
		{0x03, 0x01, 0x00, 0x0B, 0x0E},
		{0x06, 0x05, 0x04, 0x0D, 0x0F},
		{0x07, 0x11, 0x10, 0x0C, 0xFF},
		{0x0A, 0x09, 0x08, 0x02, 0xFF}
};

void kp_cb(uint8_t row, uint8_t col);
void kr_cb(uint8_t row, uint8_t col);

Keypad *kpad;
uint8_t curchar;
//uint8_t curcmd;
uint8_t nchars;
char buffer[MAX_BUFF+1];

void refresh_display(void);

void timeout_h() {
#if defined(HAS_LED)
	led = !led;
#endif
}

#ifdef DEBUG

DigitalOut dbgpin(DBGPIN);
inline void pulse(uint8_t count=1, bool stayup=false)
{
	dbgpin = 0;
	wait_us(2);
	while (count--)
	{
		dbgpin = 1;
		wait_us(2);
		dbgpin = 0;
		wait_us(2);
	}
	if (stayup)
		dbgpin = 1;
}
#else
inline void pulse(uint8_t count=1, bool stayup=false)
{}
#endif

// callbacks & thread functions
void loop();
void reset(void);
void reset_irq(void);
void qei_cb(int dir);
void end_splashscreen(void);

/*
	#if defined(HAVE_PC)
	FileHandle *mbed::mbed_override_console(int fd)
	{
	return static_cast<FileHandle*> (&pc);
	}
	#endif
*/

void setup() {
#if defined(HAVE_PC)
#if   defined(TARGET_NUCLEO_F446RE)
	pc.set_baud(115200);
#endif

/*
	#if  defined(TARGET_HP34970_FP_F303RD)
  pc.init();
  pc.connect();
	#endif
*/
#endif

	printf("\n\nSETUP\n");
	printf("  System Core Clock = %.3f MHZ\r\n",
				 (float)SystemCoreClock/1000000);

	/*
		#if defined(HAS_LED)
		printf("Attaching Led 1: %d\n", LED1);
		blinker.attach(callback(timeout_h), 0.5f);
		#endif
	*/

	hp = NULL;
#if defined(TARGET_NUCLEO_F446RE)
	printf("Serial communication pins\r\n");
	printf("     USBRX=%d\r\n", USBRX);
	printf("     USBTX=%d\r\n", USBTX);
#endif
#if defined(TARGET_HP34970_FP_F303RD)
	printf("Serial communication via USB\r\n");
#endif

	printf("Setup HP communication pins\r\n");
	printf("     HP_RX=%d\r\n", HP_RX);
	DigitalIn(HP_RX).mode(PullDown);
	printf("     HP_TX=%d\r\n", HP_TX);
	DigitalOut(HP_TX).write(1);
	printf("     HP_RST=%d\r\n", HP_RST);
	DigitalIn(HP_RST).mode(PullDown);

	printf("  setup QEI pins\r\n");
	printf("     ENC1=%d\r\n", KP_ENC1);
	DigitalIn(KP_ENC1).mode(PullDown);
	printf("     ENC2=%d\r\n", KP_ENC2);
	DigitalIn(KP_ENC2).mode(PullDown);
	qenc.attach(&qei_cb);

	printf("  setup Keypad\r\n");
	//cur_keycode.keyevent = KEY_NONE;

	shift = false;
	kpad = new Keypad(KP_NROWS, kp_rows, KP_NCOLS, kp_columns);
	printf("  attach Keypad callbacks\r\n");
	kpad->attach(&kp_cb, &kr_cb);
	printf("  start Keypad\r\n");
	kpad->start();

	printf("Setup OLED display\r\n");
	// init the LCD
	printf("     DSP_MOSI=%d\r\n", DSP_MOSI);
	printf("     DSP_MISO=%d\r\n", DSP_MISO);
	printf("     DSP_SCLK=%d\r\n", DSP_SCLK);
	printf("     DSP_CS=%d\r\n", DSP_CS);
	printf("     DSP_RST=%d\r\n", DSP_RST);
	printf("     DSP_DC=%d\r\n", DSP_DC);
	dsp = new Display(20000000, DSP_MOSI, DSP_MISO, DSP_SCLK, DSP_CS,
					  DSP_RST, DSP_DC, "SSD1322");

	//curcmd = 0xFF;
	curchar = 0;
	nchars = 0;

	for (uint8_t i=0; i<sizeof(table)/sizeof(table[0]); ++i)
		memset(table[i].buffer, 0, MAX_BUFF+1);

	printf("  display splash screen\r\n");
	dsp->show_splashscreen();

	printf("Starting LCD thread\r\n");
	tdsp.start(&refresh_display);

	//printf("Starting Event thread\r\n");
	//tloop.start(&loop);

	/*
		dsp->clrbuff();
		show(0x00, "HH:MM:\tSS\t:mmmm", 15); // main dsp
		show(0x0C, "888", 3); // channel dsp
		show(0x0A, "\xFF\xFF\xFF\xFF", 4); // all flags
	*/

	printf("Attaching timers\r\n");
	splashscreen = true;
	splashscreen_timer.attach(callback(&end_splashscreen), 2ms);
	rst.fall(&reset_irq);

	printf("SETUP DONE\r\n");
}

void end_splashscreen(void)
{
	// print is forbidden here because we are in an ISR context here
	//printf("End of splash screen CB\r\n");
	splashscreen = false;
}

void reset_irq(void)
{
	rst_delay.attach(callback(&reset), 1ms);
}

void reset(void)
{
	if (DigitalIn(HP_RST).read() == 0) {
		if (hp == NULL) {
			printf("setup HP communication handler\r\n");
			hp = new HPSerial(HP_TX, HP_RX);
		}

		printf("!! RST !! (gstate=%d, state=%d)\r\n",
					 hp->gstate(), hp->state());
		//printf("Value is ... %X\n", hp->search());
		hp->startup();
	}
}

void refresh_display(void) {
	//uint8_t mask=1;

	while(1) {
		pulse(0, true);
		if (splashscreen == false) {
			//Thread::wait(20); // give a bit of time for some more cmds
			dsp->copy_to_lcd();
		}

		/*
			if (must_refresh & mask) {
			for(uint8_t i=0; i<sizeof(zones)/sizeof(zones[0]); i++)
			if (zones[i].flag == mask) {
			dsp->copy_to_lcd(zones[i].x0/4, (zones[i].x1+3)/4,
			zones[i].y0,   zones[i].y1);
			must_refresh &= ~mask;
			break;
			}
			}
			mask = mask << 1;
			if (mask == 0) {
			mask = 1;
			}
		*/
		pulse(0, false);

		ThisThread::sleep_for(30ms);
	}
}

void loop()
{ // run over and over
	keycode_t key = {0, 0, KEY_NONE};

	unsigned int err[8];
	for (uint8_t i=0; i<8; i++)
		err[i] = 0;
	int p, pp;  // rot encoder pulse counters
	p = 0;
	pp = 0;

	while(1) {
		p = qenc.getPulses();
		if (p != pp)
		{
			dsp->locate(0, 0);
			dsp->printf("Pulses = %d  ", p);
			dsp->copy_to_lcd();
			pp = p;
		}

		if (knob != 0)
		{
			if (hp != NULL)
			{
				printf("Sending keycode %X\r\n", knob);
				hp->sendkey(knob);
				printf("   DONE\r\n");
			}
			else
			{
				dsp->locate(70, 0);
				dsp->printf("Knob = %X  ", knob);
				dsp->copy_to_lcd();
			}
			knob = 0;
		}

		if (!key_buf.empty()) //cur_keycode.keyevent != KEY_NONE)
		{
			key_buf.pop(key);
			printf("Keycode %dx%d: %s\r\n",
						 key.row, key.col, key.keyevent==KEY_PRESSED?"pressed":"released");
			if (hp != NULL) {
				uint8_t keycode = kp_mapping[key.row][key.col];
				if (key.keyevent == KEY_RELEASED)
					keycode |= 0x40;  // bit 6: key relased
				if (shift)
					keycode |= 0x20;  // bit 5: key shifted

				hp->sendkey(kp_mapping[key.row][key.col]);
			}
			else
			{
				dsp->locate(140, 0);
				dsp->printf("KC: %dx%d[0x%s%X] %s",
							key.row, key.col,
							kp_mapping[key.row][key.col] <= 0x0F ? "0" : "",
							kp_mapping[key.row][key.col],
							key.keyevent==KEY_PRESSED ? "PRE" : "REL");
				dsp->copy_to_lcd();
			}
			// cur_keycode.keyevent = KEY_NONE;
		}


		if ((hp != NULL) && (hp->cmd_available()))
		{
			HPSerial::CMD cmd;
			if (hp->pop(cmd))
			{
#if defined(HAS_LED)
				led = 1;
#endif
				for (uint8_t i=0; i<7; i++)
					if (hp->nerrors(i) > err[i]) {
						printf("ERR: %d/%d/%d/%d/%d/%d/%d\r\n",
									 hp->nerrors(0),
									 hp->nerrors(1),
									 hp->nerrors(2),
									 hp->nerrors(3),
									 hp->nerrors(4),
									 hp->nerrors(5),
									 hp->nerrors(6)
									 );
						break;
					}
				for (uint8_t i=0; i<7; i++)
					err[i] = hp->nerrors(i);

				printf("CMD[%d] %02X", (int)cmd.id, cmd.cmd);

				// 0x00: main display
				// 0x0C: channel display
				if ((cmd.cmd == 0x00) || (cmd.cmd == 0x0C))
					printf(": '%s'\r\n", cmd.value);
				else {
					printf(":");
					for (uint8_t i=0; i<cmd.size; i++)
						printf("%02x ", cmd.value[i]);
					printf("\r\n");
				}

				if (cmd.cmd == 0x01) {
          // clear a flag
					if (cmd.value[0] == 0x0E) {
            // clear the Shift flag
						shift = false;
					}
				} else if (cmd.cmd == 0x86) {
          // shutdown
					// TODO
				} else {
          // display related commands
					dsp->show(cmd.cmd, cmd.value, cmd.size);
				}
#if defined(HAS_LED)
				led = 0;
#endif
			}
		}
		//else
		ThisThread::sleep_for(1ms);
	}
}

void qei_cb(int dir)
{
	if(dir == 1) // turn right
		knob = 0x80;
	else         // turn left
		knob = 0x81;  // 83?
}

void kp_cb(uint8_t row, uint8_t col)
{
	keycode_t key;
	key.row = row;
	key.col = col;
	key.keyevent = KEY_PRESSED;
	if(!key_buf.full())
		key_buf.push(key);
}

void kr_cb(uint8_t row, uint8_t col)
{
	keycode_t key;
	key.row = row;
	key.col = col;
	key.keyevent = KEY_RELEASED;
	if(!key_buf.full())
		key_buf.push(key);
}

int main()
{
	setup();
	printf("Main loop (noop)\r\n");
	loop();
	/*
	while(1)
	{
		timeout_h();
		ThisThread::sleep_for(1ms);
	}
	*/
}
